(function baseInit(global, undefined) {
    "use strict";

    function initializeProperties(target, members) {
        var keys = Object.keys(members);
        var properties;
        var i, len;
        for (i = 0, len = keys.length; i < len; i++) {
            var key = keys[i];
            var enumerable = key.charCodeAt(0) !== /*_*/95;
            var member = members[key];
            if (member && typeof member === 'object') {
                if (member.value !== undefined || typeof member.get === 'function' || typeof member.set === 'function') {
                    if (member.enumerable === undefined) {
                        member.enumerable = enumerable;
                    }
                    properties = properties || {};
                    properties[key] = member;
                    continue;
                }
            }
            if (!enumerable) {
                properties = properties || {};
                properties[key] = { value: member, enumerable: enumerable, configurable: true, writable: true }
                continue;
            }
            target[key] = member;
        }
        if (properties) {
            Object.defineProperties(target, properties);
        }
    }

    (function (rootNamespace) {

        // Create the rootNamespace in the global namespace
        if (!global[rootNamespace]) {
            global[rootNamespace] = Object.create(Object.prototype);
        }

        // Cache the rootNamespace we just created in a local variable
        var _rootNamespace = global[rootNamespace];
        if (!_rootNamespace.Namespace) {
            _rootNamespace.Namespace = Object.create(Object.prototype);
        }

        function defineWithParent(parentNamespace, name, members) {
            /// <signature helpKeyword="PluginUtilities.Namespace.defineWithParent">
            /// <summary locid="PluginUtilities.Namespace.defineWithParent">
            /// Defines a new namespace with the specified name under the specified parent namespace.
            /// </summary>
            /// <param name="parentNamespace" type="Object" locid="PluginUtilities.Namespace.defineWithParent_p:parentNamespace">
            /// The parent namespace.
            /// </param>
            /// <param name="name" type="String" locid="PluginUtilities.Namespace.defineWithParent_p:name">
            /// The name of the new namespace.
            /// </param>
            /// <param name="members" type="Object" locid="PluginUtilities.Namespace.defineWithParent_p:members">
            /// The members of the new namespace.
            /// </param>
            /// <returns locid="PluginUtilities.Namespace.defineWithParent_returnValue">
            /// The newly-defined namespace.
            /// </returns>
            /// </signature>
            var currentNamespace = parentNamespace,
                namespaceFragments = name.split(".");

            for (var i = 0, len = namespaceFragments.length; i < len; i++) {
                var namespaceName = namespaceFragments[i];
                if (!currentNamespace[namespaceName]) {
                    Object.defineProperty(currentNamespace, namespaceName,
                        { value: {}, writable: false, enumerable: true, configurable: true }
                    );
                }
                currentNamespace = currentNamespace[namespaceName];
            }

            if (members) {
                initializeProperties(currentNamespace, members);
            }

            return currentNamespace;
        }

        function define(name, members) {
            /// <signature helpKeyword="PluginUtilities.Namespace.define">
            /// <summary locid="PluginUtilities.Namespace.define">
            /// Defines a new namespace with the specified name.
            /// </summary>
            /// <param name="name" type="String" locid="PluginUtilities.Namespace.define_p:name">
            /// The name of the namespace. This could be a dot-separated name for nested namespaces.
            /// </param>
            /// <param name="members" type="Object" locid="PluginUtilities.Namespace.define_p:members">
            /// The members of the new namespace.
            /// </param>
            /// <returns locid="PluginUtilities.Namespace.define_returnValue">
            /// The newly-defined namespace.
            /// </returns>
            /// </signature>
            return defineWithParent(global, name, members);
        }

        // Establish members of the "PluginUtilities.Namespace" namespace
        Object.defineProperties(_rootNamespace.Namespace, {

            defineWithParent: { value: defineWithParent, writable: true, enumerable: true, configurable: true },

            define: { value: define, writable: true, enumerable: true, configurable: true }

        });

    })("PluginUtilities");

    (function (PluginUtilities) {

        function define(constructor, instanceMembers, staticMembers) {
            /// <signature helpKeyword="PluginUtilities.Class.define">
            /// <summary locid="PluginUtilities.Class.define">
            /// Defines a class using the given constructor and the specified instance members.
            /// </summary>
            /// <param name="constructor" type="Function" locid="PluginUtilities.Class.define_p:constructor">
            /// A constructor function that is used to instantiate this class.
            /// </param>
            /// <param name="instanceMembers" type="Object" locid="PluginUtilities.Class.define_p:instanceMembers">
            /// The set of instance fields, properties, and methods made available on the class.
            /// </param>
            /// <param name="staticMembers" type="Object" locid="PluginUtilities.Class.define_p:staticMembers">
            /// The set of static fields, properties, and methods made available on the class.
            /// </param>
            /// <returns type="Function" locid="PluginUtilities.Class.define_returnValue">
            /// The newly-defined class.
            /// </returns>
            /// </signature>
            constructor = constructor || function () { };
            PluginUtilities.Utilities.markSupportedForProcessing(constructor);
            if (instanceMembers) {
                initializeProperties(constructor.prototype, instanceMembers);
            }
            if (staticMembers) {
                initializeProperties(constructor, staticMembers);
            }
            return constructor;
        }

        function derive(baseClass, constructor, instanceMembers, staticMembers) {
            /// <signature helpKeyword="PluginUtilities.Class.derive">
            /// <summary locid="PluginUtilities.Class.derive">
            /// Creates a sub-class based on the supplied baseClass parameter, using prototypal inheritance.
            /// </summary>
            /// <param name="baseClass" type="Function" locid="PluginUtilities.Class.derive_p:baseClass">
            /// The class to inherit from.
            /// </param>
            /// <param name="constructor" type="Function" locid="PluginUtilities.Class.derive_p:constructor">
            /// A constructor function that is used to instantiate this class.
            /// </param>
            /// <param name="instanceMembers" type="Object" locid="PluginUtilities.Class.derive_p:instanceMembers">
            /// The set of instance fields, properties, and methods to be made available on the class.
            /// </param>
            /// <param name="staticMembers" type="Object" locid="PluginUtilities.Class.derive_p:staticMembers">
            /// The set of static fields, properties, and methods to be made available on the class.
            /// </param>
            /// <returns type="Function" locid="PluginUtilities.Class.derive_returnValue">
            /// The newly-defined class.
            /// </returns>
            /// </signature>
            if (baseClass) {
                constructor = constructor || function () { };
                var basePrototype = baseClass.prototype;
                constructor.prototype = Object.create(basePrototype);
                PluginUtilities.Utilities.markSupportedForProcessing(constructor);
                Object.defineProperty(constructor.prototype, "constructor", { value: constructor, writable: true, configurable: true, enumerable: true });
                if (instanceMembers) {
                    initializeProperties(constructor.prototype, instanceMembers);
                }
                if (staticMembers) {
                    initializeProperties(constructor, staticMembers);
                }
                return constructor;
            } else {
                return define(constructor, instanceMembers, staticMembers);
            }
        }

        function mix(constructor) {
            /// <signature helpKeyword="PluginUtilities.Class.mix">
            /// <summary locid="PluginUtilities.Class.mix">
            /// Defines a class using the given constructor and the union of the set of instance members
            /// specified by all the mixin objects. The mixin parameter list is of variable length.
            /// </summary>
            /// <param name="constructor" locid="PluginUtilities.Class.mix_p:constructor">
            /// A constructor function that is used to instantiate this class.
            /// </param>
            /// <returns locid="PluginUtilities.Class.mix_returnValue">
            /// The newly-defined class.
            /// </returns>
            /// </signature>
            constructor = constructor || function () { };
            var i, len;
            for (i = 1, len = arguments.length; i < len; i++) {
                initializeProperties(constructor.prototype, arguments[i]);
            }
            return constructor;
        }

        // Establish members of "PluginUtilities.Class" namespace
        PluginUtilities.Namespace.define("PluginUtilities.Class", {
            define: define,
            derive: derive,
            mix: mix
        });

    })(PluginUtilities);

})(this);

(function baseUtilsInit(global, PluginUtilities) {
    "use strict";

    var hasWinRT = !!global.Windows;

    var strings = {
        get notSupportedForProcessing() { return PluginUtilities.Resources._getPluginUtilitiesString("base/notSupportedForProcessing").value; }
    };

    function nop(v) {
        return v;
    }

    function getMemberFiltered(name, root, filter) {
        return name.split(".").reduce(function (currentNamespace, name) {
            if (currentNamespace) {
                return filter(currentNamespace[name]);
            }
            return null;
        }, root);
    }

    // Establish members of "PluginUtilities.Utilities" namespace
    PluginUtilities.Namespace.define("PluginUtilities.Utilities", {
        // Used for mocking in tests
        _setHasWinRT: {
            value: function (value) {
                hasWinRT = value;
            },
            configurable: false,
            writable: false,
            enumerable: false
        },

        /// <field type="Boolean" locid="PluginUtilities.Utilities.hasWinRT" helpKeyword="PluginUtilities.Utilities.hasWinRT">Determine if WinRT is accessible in this script context.</field>
        hasWinRT: {
            get: function () { return hasWinRT; },
            configurable: false,
            enumerable: true
        },

        _getMemberFiltered: getMemberFiltered,

        getMember: function (name, root) {
            /// <signature helpKeyword="PluginUtilities.Utilities.getMember">
            /// <summary locid="PluginUtilities.Utilities.getMember">
            /// Gets the leaf-level type or namespace specified by the name parameter.
            /// </summary>
            /// <param name="name" locid="PluginUtilities.Utilities.getMember_p:name">
            /// The name of the member.
            /// </param>
            /// <param name="root" locid="PluginUtilities.Utilities.getMember_p:root">
            /// The root to start in. Defaults to the global object.
            /// </param>
            /// <returns locid="PluginUtilities.Utilities.getMember_returnValue">
            /// The leaf-level type or namespace in the specified parent namespace.
            /// </returns>
            /// </signature>
            if (!name) {
                return null;
            }
            return getMemberFiltered(name, root || global, nop);
        },

        ready: function (callback, async) {
            /// <signature helpKeyword="PluginUtilities.Utilities.ready">
            /// <summary locid="PluginUtilities.Utilities.ready">
            /// Ensures that the specified function executes only after the DOMContentLoaded event has fired
            /// for the current page.
            /// </summary>
            /// <returns locid="PluginUtilities.Utilities.ready_returnValue">A promise that completes after DOMContentLoaded has occurred.</returns>
            /// <param name="callback" optional="true" locid="PluginUtilities.Utilities.ready_p:callback">
            /// A function that executes after DOMContentLoaded has occurred.
            /// </param>
            /// <param name="async" optional="true" locid="PluginUtilities.Utilities.ready_p:async">
            /// If true, the callback should be executed asynchronously.
            /// </param>
            /// </signature>
            return new PluginUtilities.Promise(function (c, e) {
                function complete() {
                    if (callback) {
                        try {
                            callback();
                            c();
                        }
                        catch (err) {
                            e(err);
                        }
                    }
                    else {
                        c();
                    }
                }

                var readyState = PluginUtilities.Utilities.testReadyState;
                if (!readyState) {
                    if (global.document) {
                        readyState = document.readyState;
                    }
                    else {
                        readyState = "complete";
                    }
                }
                if (readyState === "complete" || (global.document && document.body !== null)) {
                    if (async) {
                        msSetImmediate(complete);
                    }
                    else {
                        complete();
                    }
                }
                else {
                    global.addEventListener("DOMContentLoaded", complete, false);
                }
            });
        },

        /// <field type="Boolean" locid="PluginUtilities.Utilities.strictProcessing" helpKeyword="PluginUtilities.Utilities.strictProcessing">Determines if strict declarative processing is enabled in this script context.</field>
        strictProcessing: {
            get: function () { return true; },
            configurable: false,
            enumerable: true,
        },

        markSupportedForProcessing: {
            value: function (func) {
                /// <signature helpKeyword="PluginUtilities.Utilities.markSupportedForProcessing">
                /// <summary locid="PluginUtilities.Utilities.markSupportedForProcessing">
                /// Marks a function as being compatible with declarative processing, such as PluginUtilities.UI.processAll
                /// or PluginUtilities.Binding.processAll.
                /// </summary>
                /// <param name="func" type="Function" locid="PluginUtilities.Utilities.markSupportedForProcessing_p:func">
                /// The function to be marked as compatible with declarative processing.
                /// </param>
                /// <returns locid="PluginUtilities.Utilities.markSupportedForProcessing_returnValue">
                /// The input function.
                /// </returns>
                /// </signature>
                func.supportedForProcessing = true;
                return func;
            },
            configurable: false,
            writable: false,
            enumerable: true
        },

        requireSupportedForProcessing: {
            value: function (value) {
                /// <signature helpKeyword="PluginUtilities.Utilities.requireSupportedForProcessing">
                /// <summary locid="PluginUtilities.Utilities.requireSupportedForProcessing">
                /// Asserts that the value is compatible with declarative processing, such as PluginUtilities.UI.processAll
                /// or PluginUtilities.Binding.processAll. If it is not compatible an exception will be thrown.
                /// </summary>
                /// <param name="value" type="Object" locid="PluginUtilities.Utilities.requireSupportedForProcessing_p:value">
                /// The value to be tested for compatibility with declarative processing. If the
                /// value is a function it must be marked with a property 'supportedForProcessing'
                /// with a value of true.
                /// </param>
                /// <returns locid="PluginUtilities.Utilities.requireSupportedForProcessing_returnValue">
                /// The input value.
                /// </returns>
                /// </signature>
                var supportedForProcessing = true;
                if (value === global) {
                    supportedForProcessing = false;
                } else if (value instanceof Function && !value.supportedForProcessing) {
                    supportedForProcessing = false;
                }

                if (supportedForProcessing) {
                    return value;
                }

                throw new PluginUtilities.ErrorFromName("PluginUtilities.Utilities.requireSupportedForProcessing", PluginUtilities.Resources._formatString(strings.notSupportedForProcessing, value));
            },
            configurable: false,
            writable: false,
            enumerable: true
        },

    });

    PluginUtilities.Namespace.define("PluginUtilities", {
        validation: false,

        strictProcessing: {
            value: function () {
                /// <signature helpKeyword="PluginUtilities.strictProcessing">
                /// <summary locid="PluginUtilities.strictProcessing">
                /// Strict processing is always enforced, this method has no effect.
                /// </summary>
                /// </signature>
            },
            configurable: false,
            writable: false,
            enumerable: false
        },
    });
})(this, PluginUtilities);

(function logInit() {
    "use strict";

    var spaceR = /\s+/g;
    var typeR = /^(error|warn|info|log)$/;

    function format(message, tag, type) {
        /// <signature helpKeyword="PluginUtilities.Utilities.formatLog">
        /// <summary locid="PluginUtilities.Utilities.formatLog">
        /// Adds tags and type to a logging message.
        /// </summary>
        /// <param name="message" type="String" locid="PluginUtilities.Utilities.startLog_p:message">The message to be formatted.</param>
        /// <param name="tag" type="String" locid="PluginUtilities.Utilities.startLog_p:tag">The tag(s) to be applied to the message. Multiple tags should be separated by spaces.</param>
        /// <param name="type" type="String" locid="PluginUtilities.Utilities.startLog_p:type">The type of the message.</param>
        /// <returns type="String" locid="PluginUtilities.Utilities.startLog_returnValue">The formatted message.</returns>
        /// </signature>
        var m = message;
        if (typeof (m) === "function") { m = m(); }

        return ((type && typeR.test(type)) ? ("") : (type ? (type + ": ") : "")) +
            (tag ? tag.replace(spaceR, ":") + ": " : "") +
            m;
    }
    function defAction(message, tag, type) {
        var m = PluginUtilities.Utilities.formatLog(message, tag, type);
        console[(type && typeR.test(type)) ? type : "log"](m);
    }
    function escape(s) {
        // \s (whitespace) is used as separator, so don't escape it
        return s.replace(/[-[\]{}()*+?.,\\^$|#]/g, "\\$&");
    }
    PluginUtilities.Namespace.define("PluginUtilities.Utilities", {
        startLog: function (options) {
            /// <signature helpKeyword="PluginUtilities.Utilities.startLog">
            /// <summary locid="PluginUtilities.Utilities.startLog">
            /// Configures a logger that writes messages containing the specified tags from PluginUtilities.log to console.log.
            /// </summary>
            /// <param name="options" type="String" locid="PluginUtilities.Utilities.startLog_p:options">The tags for messages to log. Multiple tags should be separated by spaces.</param>
            /// </signature>
            /// <signature>
            /// <summary locid="PluginUtilities.Utilities.startLog2">
            /// Configure a logger to write PluginUtilities.log output.
            /// </summary>
            /// <param name="options" type="Object" locid="PluginUtilities.Utilities.startLog_p:options2">
            /// May contain .type, .tags, .excludeTags and .action properties.
            /// - .type is a required tag.
            /// - .excludeTags is a space-separated list of tags, any of which will result in a message not being logged.
            /// - .tags is a space-separated list of tags, any of which will result in a message being logged.
            /// - .action is a function that, if present, will be called with the log message, tags and type. The default is to log to the console.
            /// </param>
            /// </signature>
            options = options || {};
            if (typeof options === "string") {
                options = { tags: options };
            }
            var el = options.type && new RegExp("^(" + escape(options.type).replace(spaceR, " ").split(" ").join("|") + ")$");
            var not = options.excludeTags && new RegExp("(^|\\s)(" + escape(options.excludeTags).replace(spaceR, " ").split(" ").join("|") + ")(\\s|$)", "i");
            var has = options.tags && new RegExp("(^|\\s)(" + escape(options.tags).replace(spaceR, " ").split(" ").join("|") + ")(\\s|$)", "i");
            var action = options.action || defAction;

            if (!el && !not && !has && !PluginUtilities.log) {
                PluginUtilities.log = action;
                return;
            }

            var result = function (message, tag, type) {
                if (!((el && !el.test(type))          // if the expected log level is not satisfied
                    || (not && not.test(tag))         // if any of the excluded categories exist
                    || (has && !has.test(tag)))) {    // if at least one of the included categories doesn't exist
                    action(message, tag, type);
                }

                result.next && result.next(message, tag, type);
            };
            result.next = PluginUtilities.log;
            PluginUtilities.log = result;
        },
        stopLog: function () {
            /// <signature helpKeyword="PluginUtilities.Utilities.stopLog">
            /// <summary locid="PluginUtilities.Utilities.stopLog">
            /// Removes the previously set up logger.
            /// </summary>
            /// </signature>
            delete PluginUtilities.log;
        },
        formatLog: format
    });
})();

(function eventsInit(PluginUtilities, undefined) {
    "use strict";


    function createEventProperty(name) {
        var eventPropStateName = "_on" + name + "state";

        return {
            get: function () {
                var state = this[eventPropStateName];
                return state && state.userHandler;
            },
            set: function (handler) {
                var state = this[eventPropStateName];
                if (handler) {
                    if (!state) {
                        state = { wrapper: function (evt) { return state.userHandler(evt); }, userHandler: handler };
                        Object.defineProperty(this, eventPropStateName, { value: state, enumerable: false, writable: true, configurable: true });
                        this.addEventListener(name, state.wrapper, false);
                    }
                    state.userHandler = handler;
                } else if (state) {
                    this.removeEventListener(name, state.wrapper, false);
                    this[eventPropStateName] = null;
                }
            },
            enumerable: true
        }
    }

    function createEventProperties(events) {
        /// <signature helpKeyword="PluginUtilities.Utilities.createEventProperties">
        /// <summary locid="PluginUtilities.Utilities.createEventProperties">
        /// Creates an object that has one property for each name passed to the function.
        /// </summary>
        /// <param name="events" locid="PluginUtilities.Utilities.createEventProperties_p:events">
        /// A variable list of property names.
        /// </param>
        /// <returns locid="PluginUtilities.Utilities.createEventProperties_returnValue">
        /// The object with the specified properties. The names of the properties are prefixed with 'on'.
        /// </returns>
        /// </signature>
        var props = {};
        for (var i = 0, len = arguments.length; i < len; i++) {
            var name = arguments[i];
            props["on" + name] = createEventProperty(name);
        }
        return props;
    }

    var EventMixinEvent = PluginUtilities.Class.define(
        function EventMixinEvent_ctor(type, detail, target) {
            this.detail = detail;
            this.target = target;
            this.timeStamp = Date.now();
            this.type = type;
        },
        {
            bubbles: { value: false, writable: false },
            cancelable: { value: false, writable: false },
            currentTarget: {
                get: function () { return this.target; }
            },
            defaultPrevented: {
                get: function () { return this._preventDefaultCalled; }
            },
            trusted: { value: false, writable: false },
            eventPhase: { value: 0, writable: false },
            target: null,
            timeStamp: null,
            type: null,

            preventDefault: function () {
                this._preventDefaultCalled = true;
            },
            stopImmediatePropagation: function () {
                this._stopImmediatePropagationCalled = true;
            },
            stopPropagation: function () {
            }
        }, {
            supportedForProcessing: false,
        }
    );

    var eventMixin = {
        _listeners: null,

        addEventListener: function (type, listener, useCapture) {
            /// <signature helpKeyword="PluginUtilities.Utilities.eventMixin.addEventListener">
            /// <summary locid="PluginUtilities.Utilities.eventMixin.addEventListener">
            /// Adds an event listener to the control.
            /// </summary>
            /// <param name="type" locid="PluginUtilities.Utilities.eventMixin.addEventListener_p:type">
            /// The type (name) of the event.
            /// </param>
            /// <param name="listener" locid="PluginUtilities.Utilities.eventMixin.addEventListener_p:listener">
            /// The listener to invoke when the event gets raised.
            /// </param>
            /// <param name="useCapture" locid="PluginUtilities.Utilities.eventMixin.addEventListener_p:useCapture">
            /// if true initiates capture, otherwise false.
            /// </param>
            /// </signature>
            useCapture = useCapture || false;
            this._listeners = this._listeners || {};
            var eventListeners = (this._listeners[type] = this._listeners[type] || []);
            for (var i = 0, len = eventListeners.length; i < len; i++) {
                var l = eventListeners[i];
                if (l.useCapture === useCapture && l.listener === listener) {
                    return;
                }
            }
            eventListeners.push({ listener: listener, useCapture: useCapture });
        },
        dispatchEvent: function (type, details) {
            /// <signature helpKeyword="PluginUtilities.Utilities.eventMixin.dispatchEvent">
            /// <summary locid="PluginUtilities.Utilities.eventMixin.dispatchEvent">
            /// Raises an event of the specified type and with the specified additional properties.
            /// </summary>
            /// <param name="type" locid="PluginUtilities.Utilities.eventMixin.dispatchEvent_p:type">
            /// The type (name) of the event.
            /// </param>
            /// <param name="details" locid="PluginUtilities.Utilities.eventMixin.dispatchEvent_p:details">
            /// The set of additional properties to be attached to the event object when the event is raised.
            /// </param>
            /// <returns type="Boolean" locid="PluginUtilities.Utilities.eventMixin.dispatchEvent_returnValue">
            /// true if preventDefault was called on the event.
            /// </returns>
            /// </signature>
            var listeners = this._listeners && this._listeners[type];
            if (listeners) {
                var eventValue = new EventMixinEvent(type, details, this);
                // Need to copy the array to protect against people unregistering while we are dispatching
                listeners = listeners.slice(0, listeners.length);
                for (var i = 0, len = listeners.length; i < len && !eventValue._stopImmediatePropagationCalled; i++) {
                    listeners[i].listener(eventValue);
                }
                return eventValue.defaultPrevented || false;
            }
            return false;
        },
        removeEventListener: function (type, listener, useCapture) {
            /// <signature helpKeyword="PluginUtilities.Utilities.eventMixin.removeEventListener">
            /// <summary locid="PluginUtilities.Utilities.eventMixin.removeEventListener">
            /// Removes an event listener from the control.
            /// </summary>
            /// <param name="type" locid="PluginUtilities.Utilities.eventMixin.removeEventListener_p:type">
            /// The type (name) of the event.
            /// </param>
            /// <param name="listener" locid="PluginUtilities.Utilities.eventMixin.removeEventListener_p:listener">
            /// The listener to remove.
            /// </param>
            /// <param name="useCapture" locid="PluginUtilities.Utilities.eventMixin.removeEventListener_p:useCapture">
            /// Specifies whether to initiate capture.
            /// </param>
            /// </signature>
            useCapture = useCapture || false;
            var listeners = this._listeners && this._listeners[type];
            if (listeners) {
                for (var i = 0, len = listeners.length; i < len; i++) {
                    var l = listeners[i];
                    if (l.listener === listener && l.useCapture === useCapture) {
                        listeners.splice(i, 1);
                        if (listeners.length === 0) {
                            delete this._listeners[type];
                        }
                        // Only want to remove one element for each call to removeEventListener
                        break;
                    }
                }
            }
        }
    };

    PluginUtilities.Namespace.define("PluginUtilities.Utilities", {
        _createEventProperty: createEventProperty,
        createEventProperties: createEventProperties,
        eventMixin: eventMixin
    });

})(PluginUtilities);

(function promiseInit(global, undefined) {
    "use strict";

    global.Debug && (global.Debug.setNonUserCodeExceptions = true);

    var ListenerType = PluginUtilities.Class.mix(PluginUtilities.Class.define(null, { /*empty*/ }, { supportedForProcessing: false }), PluginUtilities.Utilities.eventMixin);
    var promiseEventListeners = new ListenerType();
    // make sure there is a listeners collection so that we can do a more trivial check below
    promiseEventListeners._listeners = {};
    var errorET = "error";
    var canceledName = "Canceled";
    var tagWithStack = false;
    var tag = {
        promise: 0x01,
        thenPromise: 0x02,
        errorPromise: 0x04,
        exceptionPromise: 0x08,
        completePromise: 0x10,
    };
    tag.all = tag.promise | tag.thenPromise | tag.errorPromise | tag.exceptionPromise | tag.completePromise;

    //
    // Global error counter, for each error which enters the system we increment this once and then
    // the error number travels with the error as it traverses the tree of potential handlers.
    //
    // When someone has registered to be told about errors (PluginUtilities.Promise.callonerror) promises
    // which are in error will get tagged with a ._errorId field. This tagged field is the
    // contract by which nested promises with errors will be identified as chaining for the
    // purposes of the callonerror semantics. If a nested promise in error is encountered without
    // a ._errorId it will be assumed to be foreign and treated as an interop boundary and
    // a new error id will be minted.
    //
    var error_number = 1;

    //
    // The state machine has a interesting hiccup in it with regards to notification, in order
    // to flatten out notification and avoid recursion for synchronous completion we have an
    // explicit set of *_notify states which are responsible for notifying their entire tree
    // of children. They can do this because they know that immediate children are always
    // ThenPromise instances and we can therefore reach into their state to access the
    // _listeners collection.
    //
    // So, what happens is that a Promise will be fulfilled through the _completed or _error
    // messages at which point it will enter a *_notify state and be responsible for to move
    // its children into an (as appropriate) success or error state and also notify that child's
    // listeners of the state transition, until leaf notes are reached.
    //

    var state_created,              // -> working
        state_working,              // -> error | error_notify | success | success_notify | canceled | waiting
        state_waiting,              // -> error | error_notify | success | success_notify | waiting_canceled
        state_waiting_canceled,     // -> error | error_notify | success | success_notify | canceling
        state_canceled,             // -> error | error_notify | success | success_notify | canceling
        state_canceling,            // -> error_notify
        state_success_notify,       // -> success
        state_success,              // -> .
        state_error_notify,         // -> error
        state_error;                // -> .

    // Noop function, used in the various states to indicate that they don't support a given
    // message. Named with the somewhat cute name '_' because it reads really well in the states.

    function _() { }

    // Initial state
    //
    state_created = {
        name: "created",
        enter: function (promise) {
            promise._setState(state_working);
        },
        cancel: _,
        done: _,
        then: _,
        _completed: _,
        _error: _,
        _notify: _,
        _progress: _,
        _setCompleteValue: _,
        _setErrorValue: _
    };

    // Ready state, waiting for a message (completed/error/progress), able to be canceled
    //
    state_working = {
        name: "working",
        enter: _,
        cancel: function (promise) {
            promise._setState(state_canceled);
        },
        done: done,
        then: then,
        _completed: completed,
        _error: error,
        _notify: _,
        _progress: progress,
        _setCompleteValue: setCompleteValue,
        _setErrorValue: setErrorValue
    };

    // Waiting state, if a promise is completed with a value which is itself a promise
    // (has a then() method) it signs up to be informed when that child promise is
    // fulfilled at which point it will be fulfilled with that value.
    //
    state_waiting = {
        name: "waiting",
        enter: function (promise) {
            var waitedUpon = promise._value;
            var error = function (value) {
                if (waitedUpon._errorId) {
                    promise._chainedError(value, waitedUpon);
                } else {
                    // Because this is an interop boundary we want to indicate that this 
                    //  error has been handled by the promise infrastructure before we
                    //  begin a new handling chain.
                    //
                    callonerror(promise, value, detailsForHandledError, waitedUpon, error);
                    promise._error(value);
                }
            };
            error.handlesOnError = true;
            waitedUpon.then(
                promise._completed.bind(promise),
                error,
                promise._progress.bind(promise)
            );
        },
        cancel: function (promise) {
            promise._setState(state_waiting_canceled);
        },
        done: done,
        then: then,
        _completed: completed,
        _error: error,
        _notify: _,
        _progress: progress,
        _setCompleteValue: setCompleteValue,
        _setErrorValue: setErrorValue
    };

    // Waiting canceled state, when a promise has been in a waiting state and receives a
    // request to cancel its pending work it will forward that request to the child promise
    // and then waits to be informed of the result. This promise moves itself into the
    // canceling state but understands that the child promise may instead push it to a
    // different state.
    //
    state_waiting_canceled = {
        name: "waiting_canceled",
        enter: function (promise) {
            // Initiate a transition to canceling. Triggering a cancel on the promise
            // that we are waiting upon may result in a different state transition
            // before the state machine pump runs again.
            promise._setState(state_canceling);
            var waitedUpon = promise._value;
            if (waitedUpon.cancel) {
                waitedUpon.cancel();
            }
        },
        cancel: _,
        done: done,
        then: then,
        _completed: completed,
        _error: error,
        _notify: _,
        _progress: progress,
        _setCompleteValue: setCompleteValue,
        _setErrorValue: setErrorValue
    };

    // Canceled state, moves to the canceling state and then tells the promise to do
    // whatever it might need to do on cancelation.
    //
    state_canceled = {
        name: "canceled",
        enter: function (promise) {
            // Initiate a transition to canceling. The _cancelAction may change the state
            // before the state machine pump runs again.
            promise._setState(state_canceling);
            promise._cancelAction();
        },
        cancel: _,
        done: done,
        then: then,
        _completed: completed,
        _error: error,
        _notify: _,
        _progress: progress,
        _setCompleteValue: setCompleteValue,
        _setErrorValue: setErrorValue
    };

    // Canceling state, commits to the promise moving to an error state with an error
    // object whose 'name' and 'message' properties contain the string "Canceled"
    //
    state_canceling = {
        name: "canceling",
        enter: function (promise) {
            var error = new Error(canceledName);
            error.name = error.message;
            promise._value = error;
            promise._setState(state_error_notify);
        },
        cancel: _,
        done: _,
        then: _,
        _completed: _,
        _error: _,
        _notify: _,
        _progress: _,
        _setCompleteValue: _,
        _setErrorValue: _
    };

    // Success notify state, moves a promise to the success state and notifies all children
    //
    state_success_notify = {
        name: "complete_notify",
        enter: function (promise) {
            promise.done = CompletePromise.prototype.done;
            promise.then = CompletePromise.prototype.then;
            if (promise._listeners) {
                var queue = [promise];
                var p;
                while (queue.length) {
                    p = queue.pop();
                    p._state._notify(p, queue);
                }
            }
            promise._setState(state_success);
        },
        cancel: _,
        done: null, /*error to get here */
        then: null, /*error to get here */
        _completed: _,
        _error: _,
        _notify: notifySuccess,
        _progress: _,
        _setCompleteValue: _,
        _setErrorValue: _
    };

    // Success state, moves a promise to the success state and does NOT notify any children.
    // Some upstream promise is owning the notification pass.
    //
    state_success = {
        name: "success",
        enter: function (promise) {
            promise.done = CompletePromise.prototype.done;
            promise.then = CompletePromise.prototype.then;
            promise._cleanupAction();
        },
        cancel: _,
        done: null, /*error to get here */
        then: null, /*error to get here */
        _completed: _,
        _error: _,
        _notify: notifySuccess,
        _progress: _,
        _setCompleteValue: _,
        _setErrorValue: _
    };

    // Error notify state, moves a promise to the error state and notifies all children
    //
    state_error_notify = {
        name: "error_notify",
        enter: function (promise) {
            promise.done = ErrorPromise.prototype.done;
            promise.then = ErrorPromise.prototype.then;
            if (promise._listeners) {
                var queue = [promise];
                var p;
                while (queue.length) {
                    p = queue.pop();
                    p._state._notify(p, queue);
                }
            }
            promise._setState(state_error);
        },
        cancel: _,
        done: null, /*error to get here*/
        then: null, /*error to get here*/
        _completed: _,
        _error: _,
        _notify: notifyError,
        _progress: _,
        _setCompleteValue: _,
        _setErrorValue: _
    };

    // Error state, moves a promise to the error state and does NOT notify any children.
    // Some upstream promise is owning the notification pass.
    //
    state_error = {
        name: "error",
        enter: function (promise) {
            promise.done = ErrorPromise.prototype.done;
            promise.then = ErrorPromise.prototype.then;
            promise._cleanupAction();
        },
        cancel: _,
        done: null, /*error to get here*/
        then: null, /*error to get here*/
        _completed: _,
        _error: _,
        _notify: notifyError,
        _progress: _,
        _setCompleteValue: _,
        _setErrorValue: _
    };

    //
    // The statemachine implementation follows a very particular pattern, the states are specified
    // as static stateless bags of functions which are then indirected through the state machine
    // instance (a Promise). As such all of the functions on each state have the promise instance
    // passed to them explicitly as a parameter and the Promise instance members do a little
    // dance where they indirect through the state and insert themselves in the argument list.
    //
    // We could instead call directly through the promise states however then every caller
    // would have to remember to do things like pumping the state machine to catch state transitions.
    //

    var PromiseStateMachine = PluginUtilities.Class.define(null, {
        _listeners: null,
        _nextState: null,
        _state: null,
        _value: null,

        cancel: function () {
            /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.cancel">
            /// <summary locid="PluginUtilities.PromiseStateMachine.cancel">
            /// Attempts to cancel the fulfillment of a promised value. If the promise hasn't
            /// already been fulfilled and cancellation is supported, the promise enters
            /// the error state with a value of Error("Canceled").
            /// </summary>
            /// </signature>
            this._state.cancel(this);
            this._run();
        },
        done: function Promise_done(onComplete, onError, onProgress) {
            /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.done">
            /// <summary locid="PluginUtilities.PromiseStateMachine.done">
            /// Allows you to specify the work to be done on the fulfillment of the promised value,
            /// the error handling to be performed if the promise fails to fulfill
            /// a value, and the handling of progress notifications along the way.
            /// 
            /// After the handlers have finished executing, this function throws any error that would have been returned
            /// from then() as a promise in the error state.
            /// </summary>
            /// <param name="onComplete" type="Function" locid="PluginUtilities.PromiseStateMachine.done_p:onComplete">
            /// The function to be called if the promise is fulfilled successfully with a value.
            /// The fulfilled value is passed as the single argument. If the value is null,
            /// the fulfilled value is returned. The value returned
            /// from the function becomes the fulfilled value of the promise returned by
            /// then(). If an exception is thrown while executing the function, the promise returned
            /// by then() moves into the error state.
            /// </param>
            /// <param name="onError" type="Function" optional="true" locid="PluginUtilities.PromiseStateMachine.done_p:onError">
            /// The function to be called if the promise is fulfilled with an error. The error
            /// is passed as the single argument. If it is null, the error is forwarded.
            /// The value returned from the function is the fulfilled value of the promise returned by then().
            /// </param>
            /// <param name="onProgress" type="Function" optional="true" locid="PluginUtilities.PromiseStateMachine.done_p:onProgress">
            /// the function to be called if the promise reports progress. Data about the progress
            /// is passed as the single argument. Promises are not required to support
            /// progress.
            /// </param>
            /// </signature>
            this._state.done(this, onComplete, onError, onProgress);
        },
        then: function Promise_then(onComplete, onError, onProgress) {
            /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.then">
            /// <summary locid="PluginUtilities.PromiseStateMachine.then">
            /// Allows you to specify the work to be done on the fulfillment of the promised value,
            /// the error handling to be performed if the promise fails to fulfill
            /// a value, and the handling of progress notifications along the way.
            /// </summary>
            /// <param name="onComplete" type="Function" locid="PluginUtilities.PromiseStateMachine.then_p:onComplete">
            /// The function to be called if the promise is fulfilled successfully with a value.
            /// The value is passed as the single argument. If the value is null, the value is returned.
            /// The value returned from the function becomes the fulfilled value of the promise returned by
            /// then(). If an exception is thrown while this function is being executed, the promise returned
            /// by then() moves into the error state.
            /// </param>
            /// <param name="onError" type="Function" optional="true" locid="PluginUtilities.PromiseStateMachine.then_p:onError">
            /// The function to be called if the promise is fulfilled with an error. The error
            /// is passed as the single argument. If it is null, the error is forwarded.
            /// The value returned from the function becomes the fulfilled value of the promise returned by then().
            /// </param>
            /// <param name="onProgress" type="Function" optional="true" locid="PluginUtilities.PromiseStateMachine.then_p:onProgress">
            /// The function to be called if the promise reports progress. Data about the progress
            /// is passed as the single argument. Promises are not required to support
            /// progress.
            /// </param>
            /// <returns locid="PluginUtilities.PromiseStateMachine.then_returnValue">
            /// The promise whose value is the result of executing the complete or
            /// error function.
            /// </returns>
            /// </signature>
            return this._state.then(this, onComplete, onError, onProgress);
        },

        _chainedError: function (value, context) {
            var result = this._state._error(this, value, detailsForChainedError, context);
            this._run();
            return result;
        },
        _completed: function (value) {
            var result = this._state._completed(this, value);
            this._run();
            return result;
        },
        _error: function (value) {
            var result = this._state._error(this, value, detailsForError);
            this._run();
            return result;
        },
        _progress: function (value) {
            this._state._progress(this, value);
        },
        _setState: function (state) {
            this._nextState = state;
        },
        _setCompleteValue: function (value) {
            this._state._setCompleteValue(this, value);
            this._run();
        },
        _setChainedErrorValue: function (value, context) {
            var result = this._state._setErrorValue(this, value, detailsForChainedError, context);
            this._run();
            return result;
        },
        _setExceptionValue: function (value) {
            var result = this._state._setErrorValue(this, value, detailsForException);
            this._run();
            return result;
        },
        _run: function () {
            while (this._nextState) {
                this._state = this._nextState;
                this._nextState = null;
                this._state.enter(this);
            }
        }
    }, {
        supportedForProcessing: false
    });

    //
    // Implementations of shared state machine code.
    //

    function completed(promise, value) {
        var targetState;
        if (value && typeof value === "object" && typeof value.then === "function") {
            targetState = state_waiting;
        } else {
            targetState = state_success_notify;
        }
        promise._value = value;
        promise._setState(targetState);
    }
    function createErrorDetails(exception, error, promise, id, parent, handler) {
        return {
            exception: exception,
            error: error,
            promise: promise,
            handler: handler,
            id: id,
            parent: parent
        };
    }
    function detailsForHandledError(promise, errorValue, context, handler) {
        var exception = context._isException;
        var errorId = context._errorId;
        return createErrorDetails(
            exception ? errorValue : null,
            exception ? null : errorValue,
            promise,
            errorId,
            context,
            handler
        );
    }
    function detailsForChainedError(promise, errorValue, context) {
        var exception = context._isException;
        var errorId = context._errorId;
        setErrorInfo(promise, errorId, exception);
        return createErrorDetails(
            exception ? errorValue : null,
            exception ? null : errorValue,
            promise,
            errorId,
            context
        );
    }
    function detailsForError(promise, errorValue) {
        var errorId = ++error_number;
        setErrorInfo(promise, errorId);
        return createErrorDetails(
            null,
            errorValue,
            promise,
            errorId
        );
    }
    function detailsForException(promise, exceptionValue) {
        var errorId = ++error_number;
        setErrorInfo(promise, errorId, true);
        return createErrorDetails(
            exceptionValue,
            null,
            promise,
            errorId
        );
    }
    function done(promise, onComplete, onError, onProgress) {
        pushListener(promise, { c: onComplete, e: onError, p: onProgress });
    }
    function error(promise, value, onerrorDetails, context) {
        promise._value = value;
        callonerror(promise, value, onerrorDetails, context);
        promise._setState(state_error_notify);
    }
    function notifySuccess(promise, queue) {
        var value = promise._value;
        var listeners = promise._listeners;
        if (!listeners) {
            return;
        }
        promise._listeners = null;
        var i, len;
        for (i = 0, len = Array.isArray(listeners) ? listeners.length : 1; i < len; i++) {
            var listener = len === 1 ? listeners : listeners[i];
            var onComplete = listener.c;
            var target = listener.promise;
            if (target) {
                try {
                    target._setCompleteValue(onComplete ? onComplete(value) : value);
                } catch (ex) {
                    target._setExceptionValue(ex);
                }
                if (target._state !== state_waiting && target._listeners) {
                    queue.push(target);
                }
            } else {
                CompletePromise.prototype.done.call(promise, onComplete);
            }
        }
    }
    function notifyError(promise, queue) {
        var value = promise._value;
        var listeners = promise._listeners;
        if (!listeners) {
            return;
        }
        promise._listeners = null;
        var i, len;
        for (i = 0, len = Array.isArray(listeners) ? listeners.length : 1; i < len; i++) {
            var listener = len === 1 ? listeners : listeners[i];
            var onError = listener.e;
            var target = listener.promise;
            if (target) {
                try {
                    if (onError) {
                        if (!onError.handlesOnError) {
                            callonerror(target, value, detailsForHandledError, promise, onError);
                        }
                        target._setCompleteValue(onError(value))
                    } else {
                        target._setChainedErrorValue(value, promise);
                    }
                } catch (ex) {
                    target._setExceptionValue(ex);
                }
                if (target._state !== state_waiting && target._listeners) {
                    queue.push(target);
                }
            } else {
                ErrorPromise.prototype.done.call(promise, null, onError);
            }
        }
    }
    function callonerror(promise, value, onerrorDetailsGenerator, context, handler) {
        if (promiseEventListeners._listeners[errorET]) {
            if (value instanceof Error && value.message === canceledName) {
                return;
            }
            promiseEventListeners.dispatchEvent(errorET, onerrorDetailsGenerator(promise, value, context, handler));
        }
    }
    function progress(promise, value) {
        var listeners = promise._listeners;
        if (listeners) {
            var i, len;
            for (i = 0, len = Array.isArray(listeners) ? listeners.length : 1; i < len; i++) {
                var listener = len === 1 ? listeners : listeners[i];
                var onProgress = listener.p;
                if (onProgress) {
                    try { onProgress(value); } catch (ex) { }
                }
                if (!(listener.c || listener.e) && listener.promise) {
                    listener.promise._progress(value);
                }
            }
        }
    }
    function pushListener(promise, listener) {
        var listeners = promise._listeners;
        if (listeners) {
            // We may have either a single listener (which will never be wrapped in an array)
            // or 2+ listeners (which will be wrapped). Since we are now adding one more listener
            // we may have to wrap the single listener before adding the second.
            listeners = Array.isArray(listeners) ? listeners : [listeners];
            listeners.push(listener);
        } else {
            listeners = listener;
        }
        promise._listeners = listeners;
    }
    // The difference beween setCompleteValue()/setErrorValue() and complete()/error() is that setXXXValue() moves
    // a promise directly to the success/error state without starting another notification pass (because one
    // is already ongoing).
    function setErrorInfo(promise, errorId, isException) {
        promise._isException = isException || false;
        promise._errorId = errorId;
    }
    function setErrorValue(promise, value, onerrorDetails, context) {
        promise._value = value;
        callonerror(promise, value, onerrorDetails, context);
        promise._setState(state_error);
    }
    function setCompleteValue(promise, value) {
        var targetState;
        if (value && typeof value === "object" && typeof value.then === "function") {
            targetState = state_waiting;
        } else {
            targetState = state_success;
        }
        promise._value = value;
        promise._setState(targetState);
    }
    function then(promise, onComplete, onError, onProgress) {
        var result = new ThenPromise(promise);
        pushListener(promise, { promise: result, c: onComplete, e: onError, p: onProgress });
        return result;
    }

    //
    // Internal implementation detail promise, ThenPromise is created when a promise needs
    // to be returned from a then() method.
    //
    var ThenPromise = PluginUtilities.Class.derive(PromiseStateMachine,
        function (creator) {

            if (tagWithStack && (tagWithStack === true || (tagWithStack & tag.thenPromise))) {
                this._stack = PluginUtilities.Promise._getStack();
            }

            this._creator = creator;
            this._setState(state_created);
            this._run();
        }, {
            _creator: null,

            _cancelAction: function () { if (this._creator) { this._creator.cancel(); } },
            _cleanupAction: function () { this._creator = null; }
        }, {
            supportedForProcessing: false
        }
    );

    //
    // Slim promise implementations for already completed promises, these are created
    // under the hood on synchronous completion paths as well as by PluginUtilities.Promise.wrap
    // and PluginUtilities.Promise.wrapError.
    //

    var ErrorPromise = PluginUtilities.Class.define(
        function ErrorPromise_ctor(value) {

            if (tagWithStack && (tagWithStack === true || (tagWithStack & tag.errorPromise))) {
                this._stack = PluginUtilities.Promise._getStack();
            }

            this._value = value;
            callonerror(this, value, detailsForError);
        }, {
            cancel: function () {
                /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.cancel">
                /// <summary locid="PluginUtilities.PromiseStateMachine.cancel">
                /// Attempts to cancel the fulfillment of a promised value. If the promise hasn't
                /// already been fulfilled and cancellation is supported, the promise enters
                /// the error state with a value of Error("Canceled").
                /// </summary>
                /// </signature>
            },
            done: function ErrorPromise_done(unused, onError) {
                /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.done">
                /// <summary locid="PluginUtilities.PromiseStateMachine.done">
                /// Allows you to specify the work to be done on the fulfillment of the promised value,
                /// the error handling to be performed if the promise fails to fulfill
                /// a value, and the handling of progress notifications along the way.
                ///
                /// After the handlers have finished executing, this function throws any error that would have been returned
                /// from then() as a promise in the error state.
                /// </summary>
                /// <param name='onComplete' type='Function' locid="PluginUtilities.PromiseStateMachine.done_p:onComplete">
                /// The function to be called if the promise is fulfilled successfully with a value.
                /// The fulfilled value is passed as the single argument. If the value is null,
                /// the fulfilled value is returned. The value returned
                /// from the function becomes the fulfilled value of the promise returned by
                /// then(). If an exception is thrown while executing the function, the promise returned
                /// by then() moves into the error state.
                /// </param>
                /// <param name='onError' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.done_p:onError">
                /// The function to be called if the promise is fulfilled with an error. The error
                /// is passed as the single argument. If it is null, the error is forwarded.
                /// The value returned from the function is the fulfilled value of the promise returned by then().
                /// </param>
                /// <param name='onProgress' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.done_p:onProgress">
                /// the function to be called if the promise reports progress. Data about the progress
                /// is passed as the single argument. Promises are not required to support
                /// progress.
                /// </param>
                /// </signature>
                var value = this._value;
                if (onError) {
                    try {
                        if (!onError.handlesOnError) {
                            callonerror(null, value, detailsForHandledError, this, onError);
                        }
                        var result = onError(value);
                        if (result && typeof result === "object" && typeof result.done === "function") {
                            // If a promise is returned we need to wait on it.
                            result.done();
                        }
                        return;
                    } catch (ex) {
                        value = ex;
                    }
                }
                if (value instanceof Error && value.message === canceledName) {
                    // suppress cancel
                    return;
                }
                // force the exception to be thrown asyncronously to avoid any try/catch blocks
                //
                msSetImmediate(function () {
                    throw value;
                });
            },
            then: function ErrorPromise_then(unused, onError) {
                /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.then">
                /// <summary locid="PluginUtilities.PromiseStateMachine.then">
                /// Allows you to specify the work to be done on the fulfillment of the promised value,
                /// the error handling to be performed if the promise fails to fulfill
                /// a value, and the handling of progress notifications along the way.
                /// </summary>
                /// <param name='onComplete' type='Function' locid="PluginUtilities.PromiseStateMachine.then_p:onComplete">
                /// The function to be called if the promise is fulfilled successfully with a value.
                /// The value is passed as the single argument. If the value is null, the value is returned.
                /// The value returned from the function becomes the fulfilled value of the promise returned by
                /// then(). If an exception is thrown while this function is being executed, the promise returned
                /// by then() moves into the error state.
                /// </param>
                /// <param name='onError' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.then_p:onError">
                /// The function to be called if the promise is fulfilled with an error. The error
                /// is passed as the single argument. If it is null, the error is forwarded.
                /// The value returned from the function becomes the fulfilled value of the promise returned by then().
                /// </param>
                /// <param name='onProgress' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.then_p:onProgress">
                /// The function to be called if the promise reports progress. Data about the progress
                /// is passed as the single argument. Promises are not required to support
                /// progress.
                /// </param>
                /// <returns locid="PluginUtilities.PromiseStateMachine.then_returnValue">
                /// The promise whose value is the result of executing the complete or
                /// error function.
                /// </returns>
                /// </signature>

                // If the promise is already in a error state and no error handler is provided
                // we optimize by simply returning the promise instead of creating a new one.
                //
                if (!onError) { return this; }
                var result;
                var value = this._value;
                try {
                    if (!onError.handlesOnError) {
                        callonerror(null, value, detailsForHandledError, this, onError);
                    }
                    result = new CompletePromise(onError(value));
                } catch (ex) {
                    // If the value throw from the error handler is the same as the value
                    // provided to the error handler then there is no need for a new promise.
                    //
                    if (ex === value) {
                        result = this;
                    } else {
                        result = new ExceptionPromise(ex);
                    }
                }
                return result;
            }
        }, {
            supportedForProcessing: false
        }
    );

    var ExceptionPromise = PluginUtilities.Class.derive(ErrorPromise,
        function ExceptionPromise_ctor(value) {

            if (tagWithStack && (tagWithStack === true || (tagWithStack & tag.exceptionPromise))) {
                this._stack = PluginUtilities.Promise._getStack();
            }

            this._value = value;
            callonerror(this, value, detailsForException);
        }, {
            /* empty */
        }, {
            supportedForProcessing: false
        }
    );

    var CompletePromise = PluginUtilities.Class.define(
        function CompletePromise_ctor(value) {

            if (tagWithStack && (tagWithStack === true || (tagWithStack & tag.completePromise))) {
                this._stack = PluginUtilities.Promise._getStack();
            }

            if (value && typeof value === "object" && typeof value.then === "function") {
                var result = new ThenPromise(null);
                result._setCompleteValue(value);
                return result;
            }
            this._value = value;
        }, {
            cancel: function () {
                /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.cancel">
                /// <summary locid="PluginUtilities.PromiseStateMachine.cancel">
                /// Attempts to cancel the fulfillment of a promised value. If the promise hasn't
                /// already been fulfilled and cancellation is supported, the promise enters
                /// the error state with a value of Error("Canceled").
                /// </summary>
                /// </signature>
            },
            done: function CompletePromise_done(onComplete) {
                /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.done">
                /// <summary locid="PluginUtilities.PromiseStateMachine.done">
                /// Allows you to specify the work to be done on the fulfillment of the promised value,
                /// the error handling to be performed if the promise fails to fulfill
                /// a value, and the handling of progress notifications along the way.
                ///
                /// After the handlers have finished executing, this function throws any error that would have been returned
                /// from then() as a promise in the error state.
                /// </summary>
                /// <param name='onComplete' type='Function' locid="PluginUtilities.PromiseStateMachine.done_p:onComplete">
                /// The function to be called if the promise is fulfilled successfully with a value.
                /// The fulfilled value is passed as the single argument. If the value is null,
                /// the fulfilled value is returned. The value returned
                /// from the function becomes the fulfilled value of the promise returned by
                /// then(). If an exception is thrown while executing the function, the promise returned
                /// by then() moves into the error state.
                /// </param>
                /// <param name='onError' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.done_p:onError">
                /// The function to be called if the promise is fulfilled with an error. The error
                /// is passed as the single argument. If it is null, the error is forwarded.
                /// The value returned from the function is the fulfilled value of the promise returned by then().
                /// </param>
                /// <param name='onProgress' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.done_p:onProgress">
                /// the function to be called if the promise reports progress. Data about the progress
                /// is passed as the single argument. Promises are not required to support
                /// progress.
                /// </param>
                /// </signature>
                if (!onComplete) { return; }
                try {
                    var result = onComplete(this._value);
                    if (result && typeof result === "object" && typeof result.done === "function") {
                        result.done();
                    }
                } catch (ex) {
                    // force the exception to be thrown asynchronously to avoid any try/catch blocks
                    msSetImmediate(function () {
                        throw ex;
                    });
                }
            },
            then: function CompletePromise_then(onComplete) {
                /// <signature helpKeyword="PluginUtilities.PromiseStateMachine.then">
                /// <summary locid="PluginUtilities.PromiseStateMachine.then">
                /// Allows you to specify the work to be done on the fulfillment of the promised value,
                /// the error handling to be performed if the promise fails to fulfill
                /// a value, and the handling of progress notifications along the way.
                /// </summary>
                /// <param name='onComplete' type='Function' locid="PluginUtilities.PromiseStateMachine.then_p:onComplete">
                /// The function to be called if the promise is fulfilled successfully with a value.
                /// The value is passed as the single argument. If the value is null, the value is returned.
                /// The value returned from the function becomes the fulfilled value of the promise returned by
                /// then(). If an exception is thrown while this function is being executed, the promise returned
                /// by then() moves into the error state.
                /// </param>
                /// <param name='onError' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.then_p:onError">
                /// The function to be called if the promise is fulfilled with an error. The error
                /// is passed as the single argument. If it is null, the error is forwarded.
                /// The value returned from the function becomes the fulfilled value of the promise returned by then().
                /// </param>
                /// <param name='onProgress' type='Function' optional='true' locid="PluginUtilities.PromiseStateMachine.then_p:onProgress">
                /// The function to be called if the promise reports progress. Data about the progress
                /// is passed as the single argument. Promises are not required to support
                /// progress.
                /// </param>
                /// <returns locid="PluginUtilities.PromiseStateMachine.then_returnValue">
                /// The promise whose value is the result of executing the complete or
                /// error function.
                /// </returns>
                /// </signature>
                try {
                    // If the value returned from the completion handler is the same as the value
                    // provided to the completion handler then there is no need for a new promise.
                    //
                    var newValue = onComplete ? onComplete(this._value) : this._value;
                    return newValue === this._value ? this : new CompletePromise(newValue);
                } catch (ex) {
                    return new ExceptionPromise(ex);
                }
            }
        }, {
            supportedForProcessing: false
        }
    );

    //
    // Promise is the user-creatable PluginUtilities.Promise object.
    //

    function timeout(timeoutMS) {
        var id;
        return new PluginUtilities.Promise(
            function (c) {
                if (timeoutMS) {
                    id = setTimeout(c, timeoutMS);
                } else {
                    msSetImmediate(c);
                }
            },
            function () {
                if (id) {
                    clearTimeout(id);
                }
            }
        );
    }

    function timeoutWithPromise(timeout, promise) {
        var cancelPromise = function () { promise.cancel(); }
        var cancelTimeout = function () { timeout.cancel(); }
        timeout.then(cancelPromise);
        promise.then(cancelTimeout, cancelTimeout);
        return promise;
    }

    var staticCanceledPromise;

    var Promise = PluginUtilities.Class.derive(PromiseStateMachine,
        function Promise_ctor(init, oncancel) {
            /// <signature helpKeyword="PluginUtilities.Promise">
            /// <summary locid="PluginUtilities.Promise">
            /// A promise provides a mechanism to schedule work to be done on a value that
            /// has not yet been computed. It is a convenient abstraction for managing
            /// interactions with asynchronous APIs.
            /// </summary>
            /// <param name="init" type="Function" locid="PluginUtilities.Promise_p:init">
            /// The function that is called during construction of the  promise. The function
            /// is given three arguments (complete, error, progress). Inside this function
            /// you should add event listeners for the notifications supported by this value.
            /// </param>
            /// <param name="oncancel" optional="true" locid="PluginUtilities.Promise_p:oncancel">
            /// The function to call if a consumer of this promise wants
            /// to cancel its undone work. Promises are not required to
            /// support cancellation.
            /// </param>
            /// </signature>

            if (tagWithStack && (tagWithStack === true || (tagWithStack & tag.promise))) {
                this._stack = PluginUtilities.Promise._getStack();
            }

            this._oncancel = oncancel;
            this._setState(state_created);
            this._run();

            try {
                var complete = this._completed.bind(this);
                var error = this._error.bind(this);
                var progress = this._progress.bind(this);
                init(complete, error, progress);
            } catch (ex) {
                this._setExceptionValue(ex);
            }
        }, {
            _oncancel: null,

            _cancelAction: function () {
                if (this._oncancel) {
                    try { this._oncancel(); } catch (ex) { }
                }
            },
            _cleanupAction: function () { this._oncancel = null; }
        }, {

            addEventListener: function Promise_addEventListener(eventType, listener, capture) {
                /// <signature helpKeyword="PluginUtilities.Promise.addEventListener">
                /// <summary locid="PluginUtilities.Promise.addEventListener">
                /// Adds an event listener to the control.
                /// </summary>
                /// <param name="eventType" locid="PluginUtilities.Promise.addEventListener_p:eventType">
                /// The type (name) of the event.
                /// </param>
                /// <param name="listener" locid="PluginUtilities.Promise.addEventListener_p:listener">
                /// The listener to invoke when the event is raised.
                /// </param>
                /// <param name="capture" locid="PluginUtilities.Promise.addEventListener_p:capture">
                /// Specifies whether or not to initiate capture.
                /// </param>
                /// </signature>
                promiseEventListeners.addEventListener(eventType, listener, capture);
            },
            any: function Promise_any(values) {
                /// <signature helpKeyword="PluginUtilities.Promise.any">
                /// <summary locid="PluginUtilities.Promise.any">
                /// Returns a promise that is fulfilled when one of the input promises
                /// has been fulfilled.
                /// </summary>
                /// <param name="values" type="Array" locid="PluginUtilities.Promise.any_p:values">
                /// An array that contains promise objects or objects whose property
                /// values include promise objects.
                /// </param>
                /// <returns locid="PluginUtilities.Promise.any_returnValue">
                /// A promise that on fulfillment yields the value of the input (complete or error).
                /// </returns>
                /// </signature>
                return new Promise(
                    function (complete, error, progress) {
                        var keys = Object.keys(values);
                        var errors = Array.isArray(values) ? [] : {};
                        if (keys.length === 0) {
                            complete();
                        }
                        var canceled = 0;
                        keys.forEach(function (key) {
                            Promise.as(values[key]).then(
                                function () { complete({ key: key, value: values[key] }); },
                                function (e) {
                                    if (e instanceof Error && e.name === canceledName) {
                                        if ((++canceled) === keys.length) {
                                            complete(PluginUtilities.Promise.cancel);
                                        }
                                        return;
                                    }
                                    error({ key: key, value: values[key] });
                                }
                            );
                        });
                    },
                    function () {
                        var keys = Object.keys(values);
                        keys.forEach(function (key) {
                            var promise = Promise.as(values[key]);
                            if (typeof promise.cancel === "function") {
                                promise.cancel();
                            }
                        });
                    }
                );
            },
            as: function Promise_as(value) {
                /// <signature helpKeyword="PluginUtilities.Promise.as">
                /// <summary locid="PluginUtilities.Promise.as">
                /// Returns a promise. If the object is already a promise it is returned;
                /// otherwise the object is wrapped in a promise.
                /// </summary>
                /// <param name="value" locid="PluginUtilities.Promise.as_p:value">
                /// The value to be treated as a promise.
                /// </param>
                /// <returns locid="PluginUtilities.Promise.as_returnValue">
                /// A promise.
                /// </returns>
                /// </signature>
                if (value && typeof value === "object" && typeof value.then === "function") {
                    return value;
                }
                return new CompletePromise(value);
            },
            /// <field type="PluginUtilities.Promise" helpKeyword="PluginUtilities.Promise.cancel" locid="PluginUtilities.Promise.cancel">
            /// Canceled promise value, can be returned from a promise completion handler
            /// to indicate cancelation of the promise chain.
            /// </field>
            cancel: {
                get: function () {
                    return (staticCanceledPromise = staticCanceledPromise || new ErrorPromise(new PluginUtilities.ErrorFromName(canceledName)));
                }
            },
            dispatchEvent: function Promise_dispatchEvent(eventType, details) {
                /// <signature helpKeyword="PluginUtilities.Promise.dispatchEvent">
                /// <summary locid="PluginUtilities.Promise.dispatchEvent">
                /// Raises an event of the specified type and properties.
                /// </summary>
                /// <param name="eventType" locid="PluginUtilities.Promise.dispatchEvent_p:eventType">
                /// The type (name) of the event.
                /// </param>
                /// <param name="details" locid="PluginUtilities.Promise.dispatchEvent_p:details">
                /// The set of additional properties to be attached to the event object.
                /// </param>
                /// <returns type="Boolean" locid="PluginUtilities.Promise.dispatchEvent_returnValue">
                /// Specifies whether preventDefault was called on the event.
                /// </returns>
                /// </signature>
                return promiseEventListeners.dispatchEvent(eventType, details);
            },
            is: function Promise_is(value) {
                /// <signature helpKeyword="PluginUtilities.Promise.is">
                /// <summary locid="PluginUtilities.Promise.is">
                /// Determines whether a value fulfills the promise contract.
                /// </summary>
                /// <param name="value" locid="PluginUtilities.Promise.is_p:value">
                /// A value that may be a promise.
                /// </param>
                /// <returns type="Boolean" locid="PluginUtilities.Promise.is_returnValue">
                /// true if the specified value is a promise, otherwise false.
                /// </returns>
                /// </signature>
                return value && typeof value === "object" && typeof value.then === "function";
            },
            join: function Promise_join(values) {
                /// <signature helpKeyword="PluginUtilities.Promise.join">
                /// <summary locid="PluginUtilities.Promise.join">
                /// Creates a promise that is fulfilled when all the values are fulfilled.
                /// </summary>
                /// <param name="values" type="Object" locid="PluginUtilities.Promise.join_p:values">
                /// An object whose fields contain values, some of which may be promises.
                /// </param>
                /// <returns locid="PluginUtilities.Promise.join_returnValue">
                /// A promise whose value is an object with the same field names as those of the object in the values parameter, where
                /// each field value is the fulfilled value of a promise.
                /// </returns>
                /// </signature>
                return new Promise(
                    function (complete, error, progress) {
                        var keys = Object.keys(values);
                        var errors = Array.isArray(values) ? [] : {};
                        var results = Array.isArray(values) ? [] : {};
                        var undefineds = 0;
                        var pending = keys.length;
                        var argDone = function (key) {
                            if ((--pending) === 0) {
                                var errorCount = Object.keys(errors).length;
                                if (errorCount === 0) {
                                    complete(results);
                                } else {
                                    var canceledCount = 0;
                                    keys.forEach(function (key) {
                                        var e = errors[key];
                                        if (e instanceof Error && e.name === canceledName) {
                                            canceledCount++;
                                        }
                                    });
                                    if (canceledCount === errorCount) {
                                        complete(PluginUtilities.Promise.cancel);
                                    } else {
                                        error(errors);
                                    }
                                }
                            } else {
                                progress({ Key: key, Done: true });
                            }
                        };
                        keys.forEach(function (key) {
                            var value = values[key];
                            if (value === undefined) {
                                undefineds++;
                            } else {
                                Promise.then(value,
                                    function (value) { results[key] = value; argDone(key); },
                                    function (value) { errors[key] = value; argDone(key); }
                                );
                            }
                        });
                        pending -= undefineds;
                        if (pending === 0) {
                            complete(results);
                            return;
                        }
                    },
                    function () {
                        Object.keys(values).forEach(function (key) {
                            var promise = Promise.as(values[key]);
                            if (typeof promise.cancel === "function") {
                                promise.cancel();
                            }
                        });
                    }
                );
            },
            removeEventListener: function Promise_removeEventListener(eventType, listener, capture) {
                /// <signature helpKeyword="PluginUtilities.Promise.removeEventListener">
                /// <summary locid="PluginUtilities.Promise.removeEventListener">
                /// Removes an event listener from the control.
                /// </summary>
                /// <param name='eventType' locid="PluginUtilities.Promise.removeEventListener_eventType">
                /// The type (name) of the event.
                /// </param>
                /// <param name='listener' locid="PluginUtilities.Promise.removeEventListener_listener">
                /// The listener to remove.
                /// </param>
                /// <param name='capture' locid="PluginUtilities.Promise.removeEventListener_capture">
                /// Specifies whether or not to initiate capture.
                /// </param>
                /// </signature>
                promiseEventListeners.removeEventListener(eventType, listener, capture);
            },
            supportedForProcessing: false,
            then: function Promise_then(value, onComplete, onError, onProgress) {
                /// <signature helpKeyword="PluginUtilities.Promise.then">
                /// <summary locid="PluginUtilities.Promise.then">
                /// A static version of the promise instance method then().
                /// </summary>
                /// <param name="value" locid="PluginUtilities.Promise.then_p:value">
                /// the value to be treated as a promise.
                /// </param>
                /// <param name="onComplete" type="Function" locid="PluginUtilities.Promise.then_p:complete">
                /// The function to be called if the promise is fulfilled with a value.
                /// If it is null, the promise simply
                /// returns the value. The value is passed as the single argument.
                /// </param>
                /// <param name="onError" type="Function" optional="true" locid="PluginUtilities.Promise.then_p:error">
                /// The function to be called if the promise is fulfilled with an error. The error
                /// is passed as the single argument.
                /// </param>
                /// <param name="onProgress" type="Function" optional="true" locid="PluginUtilities.Promise.then_p:progress">
                /// The function to be called if the promise reports progress. Data about the progress
                /// is passed as the single argument. Promises are not required to support
                /// progress.
                /// </param>
                /// <returns locid="PluginUtilities.Promise.then_returnValue">
                /// A promise whose value is the result of executing the provided complete function.
                /// </returns>
                /// </signature>
                return Promise.as(value).then(onComplete, onError, onProgress);
            },
            thenEach: function Promise_thenEach(values, onComplete, onError, onProgress) {
                /// <signature helpKeyword="PluginUtilities.Promise.thenEach">
                /// <summary locid="PluginUtilities.Promise.thenEach">
                /// Performs an operation on all the input promises and returns a promise
                /// that has the shape of the input and contains the result of the operation
                /// that has been performed on each input.
                /// </summary>
                /// <param name="values" locid="PluginUtilities.Promise.thenEach_p:values">
                /// A set of values (which could be either an array or an object) of which some or all are promises.
                /// </param>
                /// <param name="onComplete" type="Function" locid="PluginUtilities.Promise.thenEach_p:complete">
                /// The function to be called if the promise is fulfilled with a value.
                /// If the value is null, the promise returns the value.
                /// The value is passed as the single argument.
                /// </param>
                /// <param name="onError" type="Function" optional="true" locid="PluginUtilities.Promise.thenEach_p:error">
                /// The function to be called if the promise is fulfilled with an error. The error
                /// is passed as the single argument.
                /// </param>
                /// <param name="onProgress" type="Function" optional="true" locid="PluginUtilities.Promise.thenEach_p:progress">
                /// The function to be called if the promise reports progress. Data about the progress
                /// is passed as the single argument. Promises are not required to support
                /// progress.
                /// </param>
                /// <returns locid="PluginUtilities.Promise.thenEach_returnValue">
                /// A promise that is the result of calling Promise.join on the values parameter.
                /// </returns>
                /// </signature>
                var result = Array.isArray(values) ? [] : {};
                Object.keys(values).forEach(function (key) {
                    result[key] = Promise.as(values[key]).then(onComplete, onError, onProgress);
                });
                return Promise.join(result);
            },
            timeout: function Promise_timeout(time, promise) {
                /// <signature helpKeyword="PluginUtilities.Promise.timeout">
                /// <summary locid="PluginUtilities.Promise.timeout">
                /// Creates a promise that is fulfilled after a timeout.
                /// </summary>
                /// <param name="timeout" type="Number" optional="true" locid="PluginUtilities.Promise.timeout_p:timeout">
                /// The timeout period in milliseconds. If this value is zero or not specified
                /// msSetImmediate is called, otherwise setTimeout is called.
                /// </param>
                /// <param name="promise" type="Promise" optional="true" locid="PluginUtilities.Promise.timeout_p:promise">
                /// A promise that will be canceled if it doesn't complete before the
                /// timeout has expired.
                /// </param>
                /// <returns type="PluginUtilities.Promise" locid="PluginUtilities.Promise.timeout_returnValue">
                /// A promise that is completed asynchronously after the specified timeout.
                /// </returns>
                /// </signature>
                var to = timeout(time);
                return promise ? timeoutWithPromise(to, promise) : to;
            },
            wrap: function Promise_wrap(value) {
                /// <signature helpKeyword="PluginUtilities.Promise.wrap">
                /// <summary locid="PluginUtilities.Promise.wrap">
                /// Wraps a non-promise value in a promise. You can use this function if you need
                /// to pass a value to a function that requires a promise.
                /// </summary>
                /// <param name="value" locid="PluginUtilities.Promise.wrap_p:value">
                /// Some non-promise value to be wrapped in a promise.
                /// </param>
                /// <returns type="PluginUtilities.Promise" locid="PluginUtilities.Promise.wrap_returnValue">
                /// A promise that is successfully fulfilled with the specified value
                /// </returns>
                /// </signature>
                return new CompletePromise(value);
            },
            wrapError: function Promise_wrapError(error) {
                /// <signature helpKeyword="PluginUtilities.Promise.wrapError">
                /// <summary locid="PluginUtilities.Promise.wrapError">
                /// Wraps a non-promise error value in a promise. You can use this function if you need
                /// to pass an error to a function that requires a promise.
                /// </summary>
                /// <param name="error" locid="PluginUtilities.Promise.wrapError_p:error">
                /// A non-promise error value to be wrapped in a promise.
                /// </param>
                /// <returns type="PluginUtilities.Promise" locid="PluginUtilities.Promise.wrapError_returnValue">
                /// A promise that is in an error state with the specified value.
                /// </returns>
                /// </signature>
                return new ErrorPromise(error);
            },

            _veryExpensiveTagWithStack: {
                get: function () { return tagWithStack; },
                set: function (value) { tagWithStack = value; }
            },
            _veryExpensiveTagWithStack_tag: tag,
            _getStack: function () {
                if (Debug.debuggerEnabled) {
                    try { throw new Error(); } catch (e) { return e.stack; }
                }
            },

        }
    );
    Object.defineProperties(Promise, PluginUtilities.Utilities.createEventProperties(errorET));

    var SignalPromise = PluginUtilities.Class.derive(PromiseStateMachine,
        function (cancel) {
            this._oncancel = cancel;
            this._setState(state_created);
            this._run();
        }, {
            _cancelAction: function () { this._oncancel && this._oncancel(); },
            _cleanupAction: function () { this._oncancel = null; }
        }, {
            supportedForProcessing: false
        }
    );

    var Signal = PluginUtilities.Class.define(
        function Signal_ctor(oncancel) {
            this._promise = new SignalPromise(oncancel);
        }, {
            promise: {
                get: function () { return this._promise; }
            },

            cancel: function Signal_cancel() {
                this._promise.cancel();
            },
            complete: function Signal_complete(value) {
                this._promise._completed(value);
            },
            error: function Signal_error(value) {
                this._promise._error(value);
            },
            progress: function Signal_progress(value) {
                this._promise._progress(value);
            }
        }, {
            supportedForProcessing: false,
        }
    );

    // Publish PluginUtilities.Promise
    //
    PluginUtilities.Namespace.define("PluginUtilities", {
        Promise: Promise,
        _Signal: Signal
    });

}(this));

// 'Plugin'.Utilities

(function eventManagerInit(PluginUtilities) {
    "use strict";

    var Event = PluginUtilities.Class.define(
          function event_ctor(type, additionalProperties, target) {
              this.target = target;
              this.timeStamp = Date.now();
              this.type = type;

              // Copy additional properties
              var eventObject = this;
              if (additionalProperties && typeof additionalProperties === "object") {
                  Object.getOwnPropertyNames(additionalProperties).forEach(function (name) {
                      var pd = Object.getOwnPropertyDescriptor(additionalProperties, name);
                      Object.defineProperty(eventObject, name, pd);
                  });
              }
          },
          {
              _preventDefaultCalled: false,
              bubbles: { value: false, writable: false },
              cancelable: { value: false, writable: false },
              currentTarget: {
                  get: function () { return this.target; }
              },
              defaultPrevented: {
                  get: function () { return this._preventDefaultCalled; }
              },
              trusted: { value: false, writable: false },
              isTrusted: { value: false, writable: false },
              eventPhase: { value: 0, writable: false },
              target: null,
              timeStamp: null,
              type: null,
              detail: null,
              preventDefault: function () {
                  this._preventDefaultCalled = true;
              },
              stopImmediatePropagation: function () {
                  this._stopImmediatePropagationCalled = true;
              },
              stopPropagation: function () {
              }
          }, {
              supportedForProcessing: false,
              NONE:0,
              CAPTURING_PHASE: 1,
              AT_TARGET: 2,
              BUBBLING_PHASE: 3
          }
      );
    
    PluginUtilities.EventManager = PluginUtilities.Class.define(
            function eventManager_ctor() {
            },
            {
                _target: null,
                _listeners: null,
                addEventListener: function (type, listener) {
                    /// <summary>
                    /// Adds an event listener.
                    /// </summary>
                    /// <param name="type" type="String">The type (name) of the event.</param>
                    /// <param name="listener" type="Function">The listener to invoke when the event gets raised.</param>

                    this._listeners = this._listeners || {};
                    var eventListeners = (this._listeners[type] = this._listeners[type] || []);
                    for (var i = 0, len = eventListeners.length; i < len; i++) {
                        var l = eventListeners[i];
                        if (l.listener === listener) {
                            return;
                        }
                    }
                    eventListeners.push({ listener: listener });
                },
                dispatchEvent: function (type, eventArg) {
                    /// <summary>
                    /// Raises an event of the specified type and with the specified additional properties.
                    /// </summary>
                    /// <param name="type" type="String">The type (name) of the event.</param>
                    /// <param name="eventArg" type="Object">The set of additional properties to be attached to the event object when the event is raised.</param>
                    /// <returns type="Boolean">true if preventDefault was called on the event.</returns>

                    var listeners = this._listeners && this._listeners[type];
                    var oneventAttribute = this._target && this._target["on" + type];
                    if (listeners || typeof oneventAttribute === "function") {
                        var eventValue = new Event(type, eventArg, this._target);
                        if (listeners) {
                            // Need to copy the array to protect against people un-registering while we are dispatching
                            listeners = listeners.slice(0, listeners.length);
                            for (var i = 0, len = listeners.length; i < len && !eventValue._stopImmediatePropagationCalled; i++) {
                                listeners[i].listener(eventValue);
                            }
                        }
                        if (typeof oneventAttribute === "function") {
                            oneventAttribute(eventValue);
                        }
                        return eventValue.defaultPrevented || false;
                    }
                    return false;
                },
                dispatchEventToListener: function (listener, type, eventArg) {
                    /// <summary>
                    /// Raises an event of the specified type to a specific listener and with the specified additional properties.
                    /// </summary>
                    /// <param name="listener" type="Function">The listener to dispatch the event to.</param>
                    /// <param name="type" type="String">The type (name) of the event.</param>
                    /// <param name="eventArg" type="Object">The set of additional properties to be attached to the event object when the event is raised.</param>
                    /// <returns type="Boolean">true if preventDefault was called on the event.</returns>

                    if (typeof listener === "function") {
                        var eventValue = new Event(type, eventArg, this._target);
                        listener(eventValue);
                    }
                },
                removeEventListener: function (type, listener) {
                    /// <summary>
                    /// Removes an event listener.
                    /// </summary>
                    /// <param name="type" type="String">The type (name) of the event.</param>
                    /// <param name="listener" type="Function">The listener to invoke when the event gets raised.</param>

                    var listeners = this._listeners && this._listeners[type];
                    if (listeners) {
                        for (var i = 0, len = listeners.length; i < len; i++) {
                            var l = listeners[i];
                            if (l.listener === listener) {
                                listeners.splice(i, 1);
                                if (listeners.length === 0) {
                                    delete this._listeners[type];
                                }
                                // Only want to remove one element for each call to removeEventListener
                                break;
                            }
                        }
                    }
                },
                setTarget: function (value) {
                    this._target = value;
                },
                eventAttached: function (type) {
                    /// <summary>
                    /// Checks if listeners are added for a certain event.
                    /// </summary>
                    /// <param name="type" type="String">The type (name) of the event.</param>
                    /// <returns type="Boolean"></returns>

                    var listeners = this._listeners && this._listeners[type];
                    return listeners && listeners.length > 0;
                }
            });

})(PluginUtilities);
(function coreInit() {
    "use strict";
    
    // Initialize the test notification global function to an empty function, if it is not already defined.
    if (!window.__n) {
        window.__n = /*@varargs */ function () {
        };
    }

    // Define the external object if not already defined
    if (!window.__hostExternalObject) {
        if (window.parent && top !== self) {
            // The control is loaded in an 'iframe'. Redirect all communication to the parent window
            var getUrlParameter = function (name) {
                var regex = new RegExp("[\\?&]" + name + "=([^&#]*)");
                var results = regex.exec(self.location.href);
                return results && results[1];
            };

            window.__hostExternalObject = {
                postMessage: function (message) {
                    return window.parent.postMessage(message, "*");
                },
                getDblClickTime: function () {
                    getUrlParameter("dblClickTime");
                },
                getScreenLeft: function (x, y) {
                    return 0;
                },
                getScreenRight: function (x, y) {
                    return self.innerWidth;
                },
                getScreenTop: function (x, y) {
                    return 0;
                },
                getScreenBottom: function (x, y) {
                    return self.innerHeight;
                },
                fireCodeMarker: function (marker) {
                    __hostExternalObject.postMessage("__" + JSON.stringify({ method: "fireCodeMarker", args: [marker] }));
                },
                reportError: function (message, url, lineNumber, additionalInfo) {
                    __hostExternalObject.postMessage("__" + JSON.stringify({ method: "reportError", args: [message, url, lineNumber, additionalInfo] }));
                },
                terminate: function () {
                    __hostExternalObject.postMessage("__" + JSON.stringify({ method: "terminate" }));
                },
                createExternalObject: function (fileAlias, clsid) {
                    return window.parent.createExternalObject(fileAlias, clsid);
                },
                hostDescription: getUrlParameter("hostDescription")
            };

            window.onmessage = function (args) {
                if (args.source === window.parent) {
                    window.__hostMessageReceived(args.data);
                }
            };
        } else if (window.external) {
            window.__hostExternalObject = {
                postMessage: function (message) {
                    return window.external.postMessage(message);
                },
                getDblClickTime: function () {
                    return window.external.getDblClickTime();
                },
                getScreenLeft: function (x, y) {
                    return window.external.getScreenLeft(x, y);
                },
                getScreenRight: function (x, y) {
                    return window.external.getScreenRight(x, y);
                },
                getScreenTop: function (x, y) {
                    return window.external.getScreenTop(x, y);
                },
                getScreenBottom: function (x, y) {
                    return window.external.getScreenBottom(x, y);
                },
                fireCodeMarker: function (marker) {
                    return window.external.fireCodeMarker(marker);
                },
                reportError: function (message, url, lineNumber, additionalInfo) {
                    return window.external.reportError(message, url, lineNumber, additionalInfo);
                },
                terminate: function () {
                    return window.external.terminate();
                },
                createExternalObject: function (fileAlias, clsid) {
                    return window.external.createExternalObject(fileAlias, clsid);
                },
                hostDescription: window.external.getHostDescription()
            };            
        } else {
            throw new Error(Plugin.Resources.getErrorString("JSPlugin.1000"));
        }
    }

    /// <var>
    /// Commands used for control messages.
    ///
    /// Note: This data structure directs serialization/deserialization of messages; it needs to stay in-sync
    ///       with its counterpart in ScriptedHostCommunicationManager.cs.
    /// </var>
    var /*@typedec*/ controlCommands = {
        /// <field>
        /// Default value for a command. Used as a place holder for messages with no control commands.
        /// </field>
        none: 0,

        /// <field>
        /// Informs the host that a new port was created.
        /// </field>
        portCreated: 1,

        /// <field>
        /// Informs the host that an existing port was closed.
        /// </field>
        portClosed: 2,

        /// <field>
        /// Informs the plug-in that a port has been connected.
        /// </field>
        portConnected: 3,

        /// <field>
        /// Informs the host that the plug-in is initialized and is ready to receive messages.
        /// </field>
        controlInitialized: 4,

        /// <field>
        /// Informs the plug-in that the host has initialized its state and that ControlReady event should be fired.
        /// </field>
        hostReady: 5,

        /// <field>
        /// Instructs the script that an event is fired.
        /// </field>
        event: 6,

        /// <field>
        /// An error occurred as a result of a previous message. This command should only occur
        /// in replies.
        /// </field>
        error: 7,

        /// <field>
        /// Informs the plug-in that the shutdown sequence has started and that the close event should be fired.
        /// </field>
        initiateShutdown: 8,

        /// <field>
        /// Informs the host that the plug-in has completed its shutdown logic and is ready to be terminated.
        /// </field>
        shutdownComplete: 9
    };

    /// <var>
    /// The default port. Represents the private communication channel between the Communication
    /// manager and its counterpart on the control side. this port is used for control commands
    /// exchange between the host and the control.
    ///
    /// Note: This is used in serialization/deserialization of messages; it needs to stay in-sync
    ///       with its counterpart in ScriptedHostCommunicationManager.js.
    /// </var>
    var defaultPort = 0;

    /// <var>
    /// Delimiter character used to separate the Scripted Host control header from the user message contents.
    /// Scripted Host headers should not include this character.
    ///
    /// Note: This data structure directs serialization/deserialization of messages; it needs to stay in-sync
    ///       with its counterpart in ScriptedHostCommunicationManager.js.
    /// </var>
    var headerDelimiter = "$";

    /// <var>
    /// The id of the last message sent from the control
    /// </var>
    var lastMessageId = 0;

    /// <var>
    /// List of messages awaiting replies
    /// </var>
    var awaitingResultList = [];

    /// <var>
    /// Event manager
    /// </var>
    var globalEventManager = new PluginUtilities.EventManager();

    /// <var>
    /// Internal local logger utility
    /// </var>
    var logger = (function () {
        var messages = [];
        var domInitialized = false;

        function logMessageLocally(message) {
            if (!domInitialized) {
                messages.push(message);
                return;
            } else {
                var messagesDiv = document.getElementById("pluginMessages");
                if (messagesDiv) {
                    messagesDiv.innerHTML += "</br>" + message;
                }
            }
        }

        // If the DOM is ready log messages directly, if not register to
        // receive the DOM load event before logging all messages.
        if (document.body) {
            domInitialized = true;
        } else {
            globalEventManager.addEventListener("load", function () {
                domInitialized = true;
                if (messages !== null) {
                    for (var i = 0; i < messages.length; i++) {
                        logMessageLocally(messages[i]);
                    }
                    messages = null;
                }
            });
        }

        return {
            log: function (message) {
                /// <summary>
                /// log a message
                /// </summary>
                /// <param name="message" type="String">The message to be logged</param>
                /// <param name="localOnly" type="Boolean" optional="true">Only use the local logger, and do not send a log message to the host</param>

                logMessageLocally(message);
            },
            logError: function (message) {
                /// <summary>
                /// log an error
                /// </summary>
                /// <param name="message" type="String">The message to be logged</param>

                this.log("Error: " + message);
            }
        };
    })();

    var Port = PluginUtilities.Class.define(function Port(name, portManager) {
        this.name = name;
        this._portManager = portManager;

        // Create an event manager
        this._eventManager = new PluginUtilities.EventManager();
        this._eventManager.setTarget(this);

        this.addEventListener = this._eventManager.addEventListener.bind(this._eventManager);
        this.removeEventListener = this._eventManager.removeEventListener.bind(this._eventManager);

        // Define state getter
        Object.defineProperty(this, "state",
            {
                get:
                  function () {
                      return this._state;
                  }
            });

        // Initially set the state to disconnected
        this._state = Port.PortState.disconnected;
    }, {
        connect: function () {
            /// <summary>
            /// Start listening on the port
            /// </summary>
            /// <returns type="Boolean"/>

            if (this._state !== Port.PortState.disconnected) {
                return;
            }

            // Register with the manager to receive events
            var port = this;
            var cookie = this._portManager.registerPort(
                this.name,
                function onConnect() {
                    if (port._state !== Port.PortState.disconnected) {
                        return;
                    }

                    // Mark the state
                    port._state = Port.PortState.connected;

                    // Fire the event
                    var eventArgs = { port: port };
                    port._eventManager.dispatchEvent("connect", eventArgs);

                },
                function onDisconnect() {
                    if (port._state !== Port.PortState.connected) {
                        return;
                    }

                    // Mark the state
                    port._state = Port.PortState.disconnected;
                },
                function onMessage(message) {
                    if (port._state !== Port.PortState.connected) {
                        return;
                    }

                    // Fire the message event
                    var eventArgs = { data: message };
                    port._eventManager.dispatchEvent("message", eventArgs);
                });

            // Remember the manager cookie
            this._cookie = cookie;
        },
        postMessage: function (message) {
            /// <summary>
            /// Post a message to the host
            /// </summary>
            /// <param name="message" type="String">Message contents</param>

            if (this._state !== Port.PortState.connected) {
                return;
            }

            this._portManager.postMessage(this._cookie, message);
        },
        sendMessage: function (message) {
            /// <summary>
            /// Sends a message to the host, and returns a promise for the reply
            /// </summary>
            /// <param name="message" type="String">Message contents</param>
            /// <returns type="PluginUtilities.Promise">Promise for the reply for the message</returns>

            if (this._state !== Port.PortState.connected) {
                return;
            }

            return this._portManager.sendMessage(this._cookie, message);
        },
        close: function () {
            /// <summary>
            /// Close the port
            /// </summary>

            if (this._state === Port.PortState.closed) {
                return;
            }

            // Set the state
            this._state = Port.PortState.closed;

            // Un-register the port
            this._portManager.unregisterPort(this._cookie);

            // Dereference the manager
            this._portManager = null;

            // Fire the event
            var eventArgs = {};
            this._eventManager.dispatchEvent("close", eventArgs);
        }
    }, {
        PortState:
        {
            connected: "connected",
            disconnected: "disconnected",
            closed: "closed"
        }
    });

    /// <var>
    /// Port manager
    /// </var>
    var portManager = (function () {
        // List of live ports
        var registeredPorts = {};

        // Lookup lists for faster access
        var portNameLookupList = {};
        var portIdLookupList = {};

        // Last live port cookie
        var lastPortIndex = 1;

        return {
            createPort: function (name) {
                /// <summary>
                /// Create a new port
                /// </summary>
                /// <param name="name" type="String">Name of the port</param>
                /// <returns type="Port"></returns>

                // Validate port name
                if (typeof name !== "string" || name.length <= 0) {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1001"));
                }

                // Do not create two ports with the same name
                if (portNameLookupList[name]) {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1002") + "\r\n" + name);
                }

                return new Port(name, this);
            },
            registerPort: function (name, onConnect, onDisconnect, onMessage) {
                if (typeof name !== "string" || name.length <= 0) {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1001"));
                }

                // Do not create two ports with the same name
                if (portNameLookupList[name]) {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1002") + "\r\n" + name);
                }

                // Validate the event handlers
                if (typeof onConnect !== "function") {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1003"));
                }

                if (typeof onDisconnect !== "function") {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1004"));
                }

                if (typeof onMessage !== "function") {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1005"));
                }

                // Assign an index for the port in the list                
                var cookie = ++lastPortIndex;

                // Add the port to the list of registered ports
                registeredPorts[cookie] = portNameLookupList[name] = {
                    id: null,
                    name: name,
                    onConnect: onConnect,
                    onDisconnect: onDisconnect,
                    onMessage: onMessage
                };

                // Let the host know about the new port
                postMessageInternal(defaultPort, controlCommands.portCreated, [name]);

                return cookie;
            },
            unregisterPort: function (cookie) {
                var entry = registeredPorts[cookie];

                if (entry) {
                    // Remove the port from the list
                    delete registeredPorts[cookie];

                    // Remove the port from lookup lists
                    if (entry.name) {
                        delete portNameLookupList[entry.name];
                    }
                    if (entry.id) {
                        delete portIdLookupList[entry.id];
                    }

                    // Let the host know that the port is closed
                    postMessageInternal(defaultPort, controlCommands.portClosed, [entry.name]);
                }
            },
            postMessage: function (cookie, message) {
                if (!registeredPorts[cookie] || registeredPorts[cookie].id === null) {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1006"));
                }
                postMessageInternal(registeredPorts[cookie].id, controlCommands.none, null, message);
            },
            sendMessage: function (cookie, message) {
                if (!registeredPorts[cookie] || registeredPorts[cookie].id === null) {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1006"));
                }
                return sendMessageInternal(registeredPorts[cookie].id, controlCommands.none, null, message);
            },
            processPortConnectedMessage: function (id, name) {
                var entry = portNameLookupList[name];
                if (entry) {
                    // Found a created port with the same name, entangle the two ports
                    entry.id = id;

                    // Add the port to the id lookup list
                    portIdLookupList[id] = entry;

                    // Fire the connect handler
                    entry.onConnect();
                } else {
                    logger.logError("JSPlugin.1010\r\n" + name);
                }
            },
            processPortClosedMessage: function (id) {
                var entry = portIdLookupList[id];
                if (entry) {
                    // Fire the disconnect handler
                    entry.onDisconnect();
                } else {
                    logger.logError("JSPlugin.1011\r\n" + id);
                }
            },
            processMessage: function (id, message) {
                var entry = portIdLookupList[id];
                if (entry) {
                    // Fire the message handler
                    entry.onMessage(message);
                } else {
                    logger.logError("JSPlugin.1012\r\n" + id);
                }
            }
        };
    })();

    function postMessageInternal(portId, command, args, payload, expectResult) {
        /// <summary>
        /// Post an message to the host
        /// </summary>
        /// <param name="portId" type="Number">The port to send the message on</param>
        /// <param name="command" type="controlCommands">The command for the message</param>
        /// <param name="args" type="String" optional="true">The command arguments</param>
        /// <param name="payload" type="String" optional="true">User message contents</param>
        /// <param name="expectResult" type="Boolean" optional="true">Replies are expected if set to true</param>

        if (lastMessageId >= Infinity) {
            lastMessageId = 0;
        }

        // Create the header
        var header = {
            msgId: ++lastMessageId,
            portId: portId,
        };

        // Set the command information
        if (command) { header.command = command; }
        if (args) { header.args = args; }

        // Set the replyRequested flag on the message
        if (expectResult) { header.replyRequested = true; }

        // Serialize the message
        var message = JSON.stringify(header);

        // Attach the user payload
        if (payload) { message += headerDelimiter + payload; }

        // If results are expected, create a promise to wrap the operation
        var result;
        if (expectResult) {
            // Create the promise to send
            result = new PluginUtilities.Promise(function (complete, error, progress) {
                awaitingResultList[header.msgId] = {
                    onComplete: complete,
                    onError: error
                };
            });
        }

        // Post the message
        __hostExternalObject.postMessage(message);

        return result;
    };

    function sendMessageInternal(portId, command, args, payload) {
        /// <summary>
        /// Post an message to the host
        /// </summary>
        /// <param name="portId" type="Number">The port to send the message on</param>
        /// <param name="command" type="controlCommands">The command for the message</param>
        /// <param name="args" type="String" optional="true">The command arguments</param>
        /// <param name="payload" type="String" optional="true">User message contents</param>
        /// <returns type="PluginUtilities.Promise"></returns>

        return postMessageInternal(portId, command, args, payload, true);
    }

    function marshalHostError(hostErrorObject) {
        var error = new Error(hostErrorObject.message + "\r\n" + hostErrorObject.stack);
        error.innerError = hostErrorObject.innerError;
        error.source = hostErrorObject.source;
        error.helpLink = hostErrorObject.helpLink;
        return error;
    }

    // Register the global message handler
    window.__hostMessageReceived = function (message) {
        if (typeof message === "string") {
            var separatorIndex = message.indexOf(headerDelimiter);

            // Check if this is a control only message (no payload)
            if (separatorIndex === -1) {
                separatorIndex = message.length;
            }

            // 'Deserialize' the message
            var header = message.substr(0, separatorIndex);
            try {
                header = JSON.parse(header);
            } catch (e) {
                logger.logError("JSPlugin.1013");
            }

            var payload = message.substr(separatorIndex + 1);

            var eventArgs;
            var i;
            var port;
            var portList;
            var portListItem;

            if (header.replyId > 0) {
                var entry = awaitingResultList[header.replyId];
                if (entry) {
                    // A reply to an earlier message, remove the entry form the list
                    delete awaitingResultList[header.replyId];

                    // Process the entry
                    switch (header.command) {
                        case controlCommands.none:
                            // The message is a reply to an earlier message. Call the promise completed event.
                            entry.onComplete(payload);
                            break;
                        case controlCommands.error:
                            // The message is a reply to an earlier message. Call the promise error event.
                            if (!header.args || !header.args.length) {
                                logger.logError("JSPlugin.1014");
                            }
                            entry.onError(header.args[0]);
                            break;
                        default:
                            // Unexpected command in the message. Terminate the promise.
                            logger.logError("JSPlugin.1015");

                            // Fire the error event to move the promise in the error state instead of waiting indefinitely
                            entry.onError(new Error(Plugin.Resources.getErrorString("JSPlugin.1015")));
                            break;
                    }
                } else if (header.command === controlCommands.error) {
                    // Error message sent as a result of a control message. Raise an error event
                    if (header.args && header.args[0]) {
                        throw marshalHostError(header.args[0]);
                    } else {
                        throw new Error(Plugin.Resources.getErrorString("JSPlugin.1007"));
                    }
                }
            } else if (header.portId > defaultPort && header.command === controlCommands.none) {
                // A user message, Pass it to the port
                portManager.processMessage(header.portId, payload);
            } else {
                // A control message received
                switch (header.command) {
                    case controlCommands.hostReady:
                        // Fire the ready event
                        globalEventManager.dispatchEvent("pluginready", {});
                        break;

                    case controlCommands.portClosed:
                        if (!header.args || !header.args.length) {
                            logger.logError("JSPlugin.1016");
                        }
                        var closedPortId = header.args[0];
                        if (typeof closedPortId !== "number") {
                            logger.logError("JSPlugin.1016");
                            return;
                        }
                        portManager.processPortClosedMessage(closedPortId);
                        break;

                    case controlCommands.portConnected:
                        if (!header.args || !header.args.length) {
                            logger.logError("JSPlugin.1017");
                        }
                        var connectedPortId = header.args[0];
                        var connectedPortName = header.args[1];
                        if (typeof connectedPortId !== "number" || typeof connectedPortName !== "string") {
                            logger.logError("JSPlugin.1017");
                            return;
                        }
                        portManager.processPortConnectedMessage(connectedPortId, connectedPortName);
                        break;

                    case controlCommands.event:
                        if (!header.args || !header.args.length) {
                            logger.logError("JSPlugin.1018");
                        }
                        var eventName = header.args[0];
                        eventArgs = header.args[1];
                        if (typeof eventName !== "string") {
                            logger.logError("JSPlugin.1019");
                        }
                        // Fire the event
                        globalEventManager.dispatchEvent(eventName, eventArgs);
                        break;

                    case controlCommands.initiateShutdown:
                        // Fire the close event
                        globalEventManager.dispatchEvent("close", eventArgs);

                        // Let the host know that the process has completed.
                        postMessageInternal(defaultPort, controlCommands.shutdownComplete);
                        break;

                    default:
                        var error;
                        if (header.args && header.args.length) {
                            error = deserializeErrorMessage(header.args[0]);
                        } else {
                            error = new Error(Plugin.Resources.getErrorString("JSPlugin.1007"));
                        }
                        throw error;
                }
            }
        }
    };

    // 'Plugin' library manager
    var libraryManager = (function () {
        return {
            attachToPublishedObject: function (name, objectDefinition, messageHandler, closeHandler, createOnFirstUse) {
                /// <param name="name" type="String">Name of the object to attach to</param>
                /// <param name="objectDefinitions" type="Object">Definitions of the marshaler</param>
                /// <param name="messageHandler" type="Function">Handle to the messages received</param>
                /// <param name="closeHandler" type="Function">Handle to the close event on the port</param>
                /// <param name="createOnFirstUse" type="Boolean" optional="true">Connect the port on the first message sent</param>
                /// <returns type="Object">A proxy of the published object</returns>

                if (typeof name !== "string") {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1008"));
                }

                if (typeof messageHandler !== "function") {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.1009"));
                }

                var interfacePortName = name;
                var interfaceObject = objectDefinition;
                var pendingMessages = [];
                var portConnectInitiated = false;

                // Force the port to be connected. This will trigger the creation of the object on the host side
                interfaceObject._forceConnect = function () {
                    if (!portConnectInitiated) {
                        // Start listening on the port
                        port.connect();
                        portConnectInitiated = true;
                    }
                };

                // Define the _postMessage function stub to collect calls and process them later
                interfaceObject._postMessage = function (message) {
                    /// <param name="message" type="String">Message contents</param>

                    // Store the message until the port is created
                    pendingMessages.push({
                        message: message,
                    });

                    // If this the first time to use this method, force connect
                    interfaceObject._forceConnect();
                };

                // Define the _sendMessage function stub to collect calls and process them later
                interfaceObject._sendMessage = function (message) {
                    /// <param name="message" type="String">Message contents</param>
                    /// <returns type="PluginUtilities.Promise">Promise for the reply of this message</returns>

                    // Create a promise for _postMessage eventually sending the message, once the port is created
                    var result = new PluginUtilities.Promise(function (complete, error, progress) {
                        // Store the message until the port is created
                        pendingMessages.push({
                            message: message,
                            onComplete: complete,
                            onError: error
                        });
                    });

                    // If this the first time to use this method, force connect
                    interfaceObject._forceConnect();

                    return result;
                };

                var port = plugin.createPort(interfacePortName);
                port.addEventListener("connect", function onConnect(e) {
                    // Remove the listener
                    port.removeEventListener("connect", onConnect);

                    // Listen on the port message event
                    port.addEventListener("message", function (eventArg) {
                        var serializedMessage = eventArg.data;
                        messageHandler(serializedMessage);
                    });

                    // Listen on the close event
                    if (typeof closeHandler === "function") {
                        port.addEventListener("close", closeHandler);
                    }

                    // Now that the port is created, define the full _postMethod implementation
                    interfaceObject._postMessage = function (message) {
                        /// <param name="message" type="String">Message contents</param>

                        return port.postMessage(message);
                    };

                    // Now that the port is created, define the full _postMethod implementation
                    interfaceObject._sendMessage = function (message) {
                        /// <param name="message" type="String">Message contents</param>
                        /// <returns type="PluginUtilities.Promise">Promise for the reply of this message</returns>

                        return port.sendMessage(message);
                    };

                    // Flush the queue of any pending messages
                    pendingMessages.forEach(function (m) {
                        if (m.onComplete) {
                            // Send message pending. We have already returned a promise for this message, so send the message, get a new promise, when it is done, complete the original promise
                            port.sendMessage(m.message)
                            .done(
                            function (callbackMessage) {
                                m.onComplete(callbackMessage);
                            },
                            function (error) {
                                m.onError(error);
                            });
                        } else {
                            // Post message pending
                            port.postMessage(m.message);
                        }
                    });
                    pendingMessages = null;
                });

                if (!createOnFirstUse) {
                    // Start listening on the port
                    interfaceObject._forceConnect();
                }

                return interfaceObject;
            }
        };
    })();

    var plugin = {
        addEventListener: globalEventManager.addEventListener.bind(globalEventManager),
        removeEventListener: globalEventManager.removeEventListener.bind(globalEventManager),
        createPort: portManager.createPort.bind(portManager),
        attachToPublishedObject: libraryManager.attachToPublishedObject.bind(libraryManager),
        _logError: logger.logError.bind(logger),
        detect: {
            isVisualStudioNativeClient: /^Microsoft Visual Studio Native Client/.test(__hostExternalObject.hostDescription)
        },
        Promise: PluginUtilities.Promise,
        Utilities: {
            EventManager: PluginUtilities.EventManager,
            marshalHostError: marshalHostError
        }
    };

    // Set the 'plugin' object as the target of all global events
    globalEventManager.setTarget(plugin);

    // Register to receive the window load event
    window.addEventListener("load", function () {
        globalEventManager.dispatchEvent("load", {});
    }, false);

    // Turn off text selection for all elements except for input and 'textarea'
    globalEventManager.addEventListener("load", function () {
        var elements = document.getElementsByTagName("*");
        for (var i = 0; i < elements.length; i++) {
            if (elements[i].nodeName === "INPUT" || elements[i].nodeName === "TEXTAREA") {
                elements[i].className += " selectElement";
            } else {
                elements[i].className += " selectNone";
            }
        }
    });

    // Notify the host that the control is initialized
    postMessageInternal(defaultPort, controlCommands.controlInitialized);

    // Publish the interface
    window.Plugin = plugin;

    // Disable the context menu supplied by IE
    document.oncontextmenu = function () {
        return false;
    };

    // Disable drag/drop behavior
    document.ondragstart = function () {
        return false;
    };

})();

// 'Plugin'.'JSONMarshaler'
(function jsonMarshalerInit() {

    Plugin.Utilities.JSONMarshaler = {
        attachToPublishedObject: function (name, objectDefinition, createOnFirstUse) {
            /// <param name="name" type="String">Name of the published object</param>
            /// <param name="objectDefinition" type="Object">Marshaler definitions</param>
            /// <param name="createOnFirstUse" type="Boolean" optional="true">Connect the port on the first message sent</param>

            // Create an event manager for this object
            var eventManager = new Plugin.Utilities.EventManager();

            // Attach to the object
            var interfaceObject = Plugin.attachToPublishedObject(name, objectDefinition,
            function onMessage(serializedMessage) {
                if (typeof serializedMessage === "string") {
                    // Handle events
                    var message = JSON.parse(serializedMessage);
                    if (typeof message.eventName === "string") {
                        // Fire the event
                        eventManager.dispatchEvent(message.eventName, message.arg);
                    } else {
                        Plugin._logError("JSPlugin.2000");
                    }
                } else {
                    Plugin._logError("JSPlugin.2001");
                }
            },
            function onClose(error) {
                Plugin._logError("JSPlugin.2002\r\n" + name);
            },
            createOnFirstUse);

            // Set the new interface object as a target for all dispatched events
            eventManager.setTarget(interfaceObject);

            // Define the _post implementation
            interfaceObject._post = function (name) {
                /// <summary>
                /// Call a method on an object exposed using JsonPortMarshaler
                /// </summary>
                /// <param name="name" type="String">Name of the method</param>
                /// <param name="..." optional="true">Arguments of the call</param>

                var message = {
                    method: name,
                    args: arguments.length > 1 ? Array.prototype.slice.call(arguments, 1) : undefined
                };

                // Post the message
                this._postMessage(JSON.stringify(message));
            };

            // Define the _call implementation
            interfaceObject._call = function (name) {
                /// <summary>
                /// Call a method on an object exposed using JsonPortMarshaler
                /// </summary>
                /// <param name="name" type="String">Name of the method</param>
                /// <param name="..." optional="true">Arguments of the call</param>
                /// <returns type="PluginUtilities.Promise">Promise for the call result</returns>

                var message = {
                    method: name,
                    args: arguments.length > 1 ? Array.prototype.slice.call(arguments, 1) : undefined
                };

                // Send the message
                return this._sendMessage(JSON.stringify(message))
                .then(
                function (responseText) {
                    var response = JSON.parse(responseText);
                    return response.result;
                });
            };

            // Expose the EventManager methods
            if (createOnFirstUse) {
                interfaceObject.addEventListener = function (type, listener) {
                    /// <summary>
                    /// Adds an event listener.
                    /// </summary>
                    /// <param name="type" type="String">The type (name) of the event.</param>
                    /// <param name="listener" type="Function">The listener to invoke when the event gets raised.</param>

                    // If this is the first use of the interface, force connecting to the host
                    interfaceObject._forceConnect();

                    eventManager.addEventListener(type, listener);

                    // Flip back to the normal behavior
                    interfaceObject.addEventListener = eventManager.addEventListener.bind(eventManager);
                };
            } else {
                interfaceObject.addEventListener = eventManager.addEventListener.bind(eventManager);
            }
            interfaceObject.removeEventListener = eventManager.removeEventListener.bind(eventManager);

            return interfaceObject;
        }
    };
})();


// Initializes the proxy object for 'Plugin'.'Output'
(function outputProxyInit() {
    "use strict";
    // Attach to the 'Plugin'.Output host object
    Plugin._outputProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.Output", {
        log: function (message) {
            this._post("log", message);
        }
    });

})();

// 'Plugin' logging
(function outputInit() {
    "use strict";

    function formatString(message, optionalParams) {
        var currentParameterIndex = 1;
        var currentSubstringIndex = 0;
        var result = "";

        // Force the message to a string
        message = "" + message;
        while (currentSubstringIndex < message.length) {
            var replacementIndex = message.indexOf("%", currentSubstringIndex);
            // Add the previous message portion
            if (replacementIndex === -1 || replacementIndex === message.length - 1) {
                result += message.substring(currentSubstringIndex);
                currentSubstringIndex = message.length;
            } else {
                result += message.substring(currentSubstringIndex, replacementIndex);
                currentSubstringIndex = replacementIndex + 1;
                var argumentValue = arguments[currentParameterIndex];
                switch (message[currentSubstringIndex]) {
                    case "d":
                    case "i":
                        if (typeof argumentValue !== "undefined") {
                            argumentValue = parseInt(argumentValue) || 0;
                        }
                        result += argumentValue;
                        currentParameterIndex++;
                        currentSubstringIndex++;
                        break;
                    case "f":
                        if (argumentValue === null) {
                            argumentValue = 0;
                        } else if (typeof argumentValue !== "undefined") {
                            argumentValue = parseFloat(argumentValue);
                        }
                        result += argumentValue;
                        currentParameterIndex++;
                        currentSubstringIndex++;
                        break;
                    case "s":
                    case "o":
                        if (typeof argumentValue !== "undefined") {
                            argumentValue = "" + argumentValue;
                        }
                        result += argumentValue;
                        currentParameterIndex++;
                        currentSubstringIndex++;
                        break;
                    case "%":
                        // Escape for %
                        result += "%";
                        currentSubstringIndex++;
                        break;
                    default:
                        // An invalid escape sequence, ignore only one character
                        result += "%";
                        break;
                }
            }
        }

        // Append any remaining parameters
        for (var i = currentParameterIndex; i < arguments.length; i++) {
            result += arguments[i];
        }

        return result;
    }

    Plugin.log = function (message, optionalParams) {
        /// <summary> log message to the host </summary>
        /// <param name='message' type='String' optional='true' />
        /// <param name='optionalParams' type='Object' optional='true'  />

        Plugin._outputProxy.log(formatString.apply(this, arguments));
    };
})();
// 'Plugin'.VS.Culture
(function cultureInit() {
    "use strict";

    // Verify the host is Visual Studio Native Client
    if (!Plugin.detect.isVisualStudioNativeClient) {
        return;
    }

    Plugin.VS = Plugin.VS || {};

    // Attach to the 'Plugin'.VS.Culture host object
    Plugin.VS._cultureProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.VS.Culture", {});

    // Private variables 
    var dir = "";
    var lang = "";
    var DateTimeFormat = [];
    var NumberFormat = [];
    var domInitialized = false;
   
    var eventManager = new Plugin.Utilities.EventManager();
    eventManager.setTarget(Plugin.VS._cultureProxy);

    // Register to receive the culture initialize event
    Plugin.VS._cultureProxy.addEventListener("cultureinitialize", function (eventArgs) {
        // If the DOM is ready set the culture attributes, if not
        // register to receive the DOM load event.
        if (!setCultureInfoAndAttributes(eventArgs.language, eventArgs.direction, eventArgs.dateTimeFormat, eventArgs.numberFormat)) {
            Plugin.addEventListener("load", function () {
                setCultureInfoAndAttributes(eventArgs.language, eventArgs.direction, eventArgs.dateTimeFormat, eventArgs.numberFormat);
            });
        }
    });

    // Register to receive the culture changed event
    Plugin.VS._cultureProxy.addEventListener("culturechanged", function (eventArgs) {
        setCultureInfoAndAttributes(eventArgs.language, eventArgs.direction, eventArgs.dateTimeFormat, eventArgs.numberFormat);
        eventManager.dispatchEvent("culturechanged");
    });

    function setCultureInfoAndAttributes(language, direction, dateTimeFormat, numberFormat) {
        /// <summary>   
        /// Set culture information and the lang and dir culture attributes on the html tag
        /// </summary>
        /// <param name="language" type="String">HTML language</param>
        /// <param name="direction" type="String">HTML language direction</param>
        /// <param name="dateTimeFormat" type="Object">Date and time information for the current culture</param>
        /// <param name="numberFormat" type="Object">Number format information for the current culture</param>
        /// <returns type="bool">True if the culture attributes are set, false otherwise</returns>

        lang = language;
        dir = direction;
        DateTimeFormat = dateTimeFormat;
        NumberFormat = numberFormat;

        // Only set the culture attributes once
        if (!domInitialized) {
            var htmlTags = document.getElementsByTagName("html");
            if (htmlTags.length > 0) {
                domInitialized = true;
                htmlTags[0].dir = dir;
                htmlTags[0].lang = lang;
                eventManager.dispatchEvent("cultureinitialize");
            } else {
                return false;
            }
        }
        return true;
    }

    // Return public functions
    Plugin.VS.Culture = {
        addEventListener: eventManager.addEventListener.bind(eventManager),
        removeEventListener: eventManager.removeEventListener.bind(eventManager)
    };

    // Return public properties
    Object.defineProperty(Plugin.VS.Culture, "dir",
       {
           get:
             function () {
                 return dir;
             }
       });

    Object.defineProperty(Plugin.VS.Culture, "lang",
        {
            get:
              function () {
                  return lang;
              }
        });

    Object.defineProperty(Plugin.VS.Culture, "DateTimeFormat",
        {
            get:
              function () {
                  return DateTimeFormat;
              }
        });

    Object.defineProperty(Plugin.VS.Culture, "NumberFormat",
        {
            get:
              function () {
                  return NumberFormat;
              }
        });
})();


// 'Plugin'.Resources
(function resourcesInit() {
    "use strict";

    // Attach to the 'Plugin'.Resources host object
    var resourcesProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.Resources", {});

    // Private variables 
    var defaultAlias = "Resources";

    // Generic error string - replaced by the culture specific error string on initialization
    var error = "An error has occurred.  Please try the operation again.  You can search for the error online: ";

    // JSON resource map that gets set when a resources changed event is fired from the host
    var resourceMap = [];

    // Regular expression used for formatting the string
    // Matches the following: '{{' , '{0}' , '}}'  and un-escaped '{' and '}'.
    var formatRegEx = /\{{2}|\{(\d+)\}|\}{2}|\{|\}/g;

    function processResourceChangeEvent(eventArgs) {
        // Set the 'plugin' generic error string
        if (typeof eventArgs.GenericError !== "string" || eventArgs.GenericError === "") {
            throw new Error(Plugin.Resources.getErrorString("JSPlugin.3000"));
        }
        error = eventArgs.GenericError;

        // Set the 'plugin' resource map
        var resources = eventArgs.ResourceMap;
        if (!resources) {
            Plugin._logError("JSPlugin.3001");
            return;
        }

        resourceMap = resources;

        // Set the default resource alias
        var defaultResource = eventArgs.DefaultAlias;
        if (defaultResource) {
            defaultAlias = defaultResource;
        }
    }

    // Register to receive the resources initialized event
    resourcesProxy.addEventListener("resourcesinitialized", processResourceChangeEvent);

    // Register to receive the resources change event
    resourcesProxy.addEventListener("resourceschanged", processResourceChangeEvent);

    function format(format, args) {
        /// <summary>
        /// Replaces the format item in a specified string with the text equivalent of the value of a corresponding Object instance in a specified array. 
        /// <param name="format" type="String">A string containing zero or more format items.</param>
        /// <param name="args" type="object" optional="true">An argument array containing zero or more objects to format.</param>
        /// </summary>
        /// <returns type="String">A formatted string</returns>

        return format.replace(formatRegEx, function (match, index) {
            var replacer;

            // Process escaped braces, get the replacement argument 
            // or throw an error for any un-escaped single braces.
            switch (match) {
                case "{{":
                    replacer = "{";
                    break;
                case "}}":
                    replacer = "}";
                    break;
                case "{":
                case "}":
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.3002"));
                default:
                    var argsIndex = parseInt(index) + 1;
                    if (args && args.length - 1 >= argsIndex) {
                        replacer = args[argsIndex];
                    } else {
                        throw new Error(Plugin.Resources.getErrorString("JSPlugin.3003"));
                    }
                    break;
            }

            // If null or undefined replace with string empty
            if (typeof replacer === "undefined" || replacer === null) {
                replacer = "";
            }

            // Convert the 'replacer' to type string
            if (typeof replacer !== "string") {
                replacer = replacer.toString();
            }

            return replacer;
        });
    }

    // Return public functions
    Plugin.Resources = {
        getString: function (resourceId) {
            /// <summary>
            /// Retrieves the resource value for a given resource id (or the formatted resource value if arguments are provided)
            /// </summary>
            /// <param name="resourceId" type="String">The resource id</param>
            /// <returns type="String">The resource value (or formatted resource value if arguments are provided)</returns>

            if (typeof resourceId !== "string" || resourceId === "") {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.3004"));
            }

            var fileName = defaultAlias;
            var key = "";
            var value = "";

            // Resource ids are constructed as follows: /ResourceFileName/Key
            // If the resource file name is omitted the default file name alias is used.
            var idParts = resourceId.split("/");

            switch (idParts.length) {
                case 1:
                    key = idParts[0];
                    break;
                case 3:
                    fileName = idParts[1];
                    key = idParts[2];
                    break;
                default:
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.3004"));
            }

            if (!resourceMap[fileName] || !resourceMap[fileName][key]) {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.3005"));
            }

            value = resourceMap[fileName][key];

            // If arguments are provided format the resource value
            // arguments[1 ... 'n'] are the replacement arguments.
            if (arguments.length > 1) {
                value = format(value, arguments);
            }

            return value;
        },

        getErrorString: function (errorId) {
            /// <summary>
            /// Retrieves a generic error string with the specific error id 
            /// </summary>
            /// <param name="errorId" type="String">The error id</param>
            /// <returns type="String">The generic error string value</returns>

            if (typeof errorId !== "string" || errorId === "") {
                throw new Error(error + "JSPlugin.3006");
            }

            return error + errorId;
        }
    };
})();


// 'Plugin'.'VS'.'Commands'
(function commandsInit() {
    "use strict";

    // Verify the host is Visual Studio Native Client
    if (!Plugin.detect.isVisualStudioNativeClient) {
        return;
    }

    Plugin.VS = Plugin.VS || {};

    // Attach to the'Plugin'.'Commands' host object	
    Plugin.VS._commandsProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.VS.Commands", {
        setCommandsStates: function (commands) {
            return this._call("setCommandsStates", commands);
        },
        showContextMenu: function (menuName, xPosition, yPosition) {
            return this._call("showContextMenu", menuName, xPosition, yPosition);
        }
    });

    var menuAliases, commandAliases;

    Plugin.VS._commandsProxy.addEventListener("commandsinitialized", function (e) {
        menuAliases = e.menuAliases;
        commandAliases = e.commandAliases;
    });

    var CommandStateMarshaler = PluginUtilities.Class.define(function (name) {
        this.name = name;
        this.enabled = null;
        this.visible = null;
    });

    // A binding to a host defined command.  Allows the user to get/set command state, and handle command execution
    var CommandBinding = PluginUtilities.Class.define(function (name, onexecute, enabled, visible) {
        this._name = name;                  // Name of the command
        this._onexecute = onexecute;    // Function which is called when the command should be executed
        this._enabled = enabled;            // Boolean which is true if the command is enabled, false otherwise
        this._visible = visible;            // Boolean which is true if the command is visible, false otherwise
    },
    {
        setState: function (state) {
            /// <summary>
            /// Sets the state for a command.
            /// </summary>
            /// <param name="state" type="Object">The state for the command in the following format:<br/>
            /// enabled: True if the command is enabled, false otherwise.<br/>
            /// visible: True if the command is visible, false otherwise.<br/>
            /// Note that all states are optional, if you don't specify a state it will not be updated for the command
            /// </param>
            var needToSetCommandStates = false;
            var commandStateMarshaler = new CommandStateMarshaler(this._name);

            if (state.hasOwnProperty("enabled") && state.enabled !== undefined) {
                this._enabled = state.enabled;
                commandStateMarshaler.enabled = state.enabled;
                needToSetCommandStates = true;
            }
            if (state.hasOwnProperty("visible") && state.visible !== undefined) {
                this._visible = state.visible;
                commandStateMarshaler.visible = state.visible;
                needToSetCommandStates = true;
            }
            if (needToSetCommandStates) {
                Plugin.VS._commandsProxy.setCommandsStates([commandStateMarshaler]);
            }
        }
    });

    // A binding to a host defined context menu.  Allows the user to show the context menu
    var ContextMenuBinding = PluginUtilities.Class.define(function (name) {
        this._name = name;  // Name of the context menu
    },
    {
        show: function (xPosition, yPosition) {
            /// <summary>
            /// Shows the context menu
            /// </summary>
            /// <param name="xPosition" type="Number">The x position in terms of screen coordinates to display the context menu</param>
            /// <param name="yPosition" type="Number">The x position in terms of screen coordinates to display the context menu</param>			

            return Plugin.VS._commandsProxy.showContextMenu(this._name, xPosition, yPosition);
        }
    });

    // Private variables
    // List of bound commands.
    var commandBindings = [];

    // Register to receive the CommandExec event
    Plugin.VS._commandsProxy.addEventListener("commandexec", function (eventArgs) {
        var commandName = eventArgs.CommandName;

        // Find the corresponding command bindings and execute the command
        if (commandBindings.hasOwnProperty(commandName)) {
            commandBindings[commandName]._onexecute();
        }
    });

    // Return public functions
    Plugin.VS.Commands = {
        bindContextMenu: function (name) {
            /// <summary>
            /// Creates a context menu binding for a host defined context menu
            /// </summary>
            /// <param name="name" type="String">The name of the context menu</param>
            /// <returns value="ContextMenuBinding">The context menu binding.</returns>
            if (!menuAliases || menuAliases.indexOf(name) === -1) {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.5000"));
            }
            return new ContextMenuBinding(name);
        },
        bindCommand: function (command) {
            /// <summary>
            /// Creates a command binding for a host defined command
            /// </summary>
            /// <param name="command" type="Object">The command specifications in the following format:<br/>
            /// name: The name of the command<br/>
            /// onexecute: The function to call when the command is executed by the host<br/>
            /// enabled: (Optional) True if the command is enabled, false otherwise.<br/>
            /// visible: (Optional) True if the command is visible, false otherwise.
            /// </param>
            /// <returns value="CommandBinding">The command binding.</returns>

            var isEnabled;
            var isVisible;
            var needToSetCommandState = false;

            if (!command.hasOwnProperty("name")) {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.5001"));
            }

            if (!commandAliases || commandAliases.indexOf(command.name) === -1) {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.5002"));
            }

            if (!command.hasOwnProperty("onexecute") || typeof command.onexecute !== "function") {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.5003"));
            }

            if (command.hasOwnProperty("enabled")) {
                isEnabled = !!command.enabled;
                needToSetCommandState = true;
            }

            if (command.hasOwnProperty("visible")) {
                isVisible = !!command.visible;
                needToSetCommandState = true;
            }

            // Verify that we haven't already constructed this command binding
            if (commandBindings.hasOwnProperty(command.name)) {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.5004"));
            }

            var newBinding = new CommandBinding(command.name, command.onexecute, isEnabled, isVisible);
            commandBindings[command.name] = newBinding;

            if (needToSetCommandState) {
                newBinding.setState({
                    enabled: isEnabled,
                    visible: isVisible
                });
            }
            return newBinding;
        },
        setStates: function () {
            /// <signature>
            /// <summary>
            /// Sets the states for multiple command bindings
            /// </summary>
            /// <param name="arguments" type="Object">The command binding for which to set the state, in the following format:<br/>
            /// command: The CommandBinding for which to set the state.<br/>
            /// enabled: True if the command is enabled, false otherwise.<br/>
            /// visible: True if the command is visible, false otherwise.<br/>
            /// Note that all states are optional, if you don't specify a state it will not be updated for the command
            /// </param>
            /// </signature>

            var commandStateMarshalers = [];

            for (var i = 0; i < arguments.length; i++) {
                var commandInstance = arguments[i];
                if (commandInstance.hasOwnProperty("command") && !!commandInstance.command && (commandInstance.command instanceof CommandBinding)) {
                    var commandStateMarshaler = new CommandStateMarshaler(commandInstance.command._name);

                    if (commandInstance.hasOwnProperty("enabled") && commandInstance.enabled !== undefined) {
                        commandStateMarshaler.enabled = commandInstance.enabled;
                    }

                    if (commandInstance.hasOwnProperty("visible") && commandInstance.visible !== undefined) {
                        commandStateMarshaler.visible = commandInstance.visible;
                    }

                    commandStateMarshalers.push(commandStateMarshaler);
                } else {
                    throw new Error(Plugin.Resources.getErrorString("JSPlugin.5005"));
                }
            }

            if (commandStateMarshalers.length > 0) {
                Plugin.VS._commandsProxy.setCommandsStates(commandStateMarshalers);

                // Now that the call was made, set the internal states
                for (i = 0; i < arguments.length; i++) {
                    commandInstance = arguments[i];

                    if (commandInstance.hasOwnProperty("enabled") && commandInstance.enabled !== undefined) {
                        commandInstance.command._enabled = commandInstance.enabled;
                    }

                    if (commandInstance.hasOwnProperty("visible") && commandInstance.visible !== undefined) {
                        commandInstance.command._visible = commandInstance.visible;
                    }
                }

            }
        }
    };

})();


    // 'Plugin'.Theme
(function themeInit() {
    "use strict";

    // Attach to the 'Plugin'.Theme host object
    var themeProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.Theme", {
        getCssFile: function (fileName) {
            /// <param name='fileName' type='String'/>

            return this._call("getCssFile", fileName);
        },
        fireThemeReady: function () {
            this._post("fireThemeReady");
        }
    });

    // Private variables
    var domInitialized = false;

    //  Indicates whether this is the initial theme changed event. (Set when a theme change is fired from the host.)
    var isInitial = false;

    /// JSON token map that gets set when a theme change event is fired from the host
    /// Token map property names match the token keys in plugin.css
    /// Example: tokenMap["plugin-scrollbar-background-color"] returns the value for the {plugin-scrollbar-background-color} token
    var tokenMap = [];

    // The base 'plugin' css that gets set when the theme is initialized
    var pluginCss;

    // These 'regex' match css declaration property name and value pairs using capture groups
    var declarationRegEx = /^(\s*)([\w\-]+)\s*:([^;^\{\*]+);\s*\/\*\s*\[([^\[\]]+)\]\s*\*\/(.*)$/gm;
    // Token 'replacer' - allow any word character or dash to match in a token name.  Also allow
    // extra opening and closing parenthesis to match so that these can be error handled correctly.
    var tokenRegEx = /\{\s*([\{\}\w\-]*)\s*\}/gm;
    var undefinedRegEx = /undefined|null/;

    var eventManager = new Plugin.Utilities.EventManager();
    eventManager.setTarget(themeProxy);

    // Register to receive the on theme initialize event
    themeProxy.addEventListener("themeinitialize", function (eventArgs) {
        pluginCss = eventArgs.PluginCss;
        if (!pluginCss) {
            Plugin._logError("JSPlugin.4000");
            return;
        }
        updateTheme(eventArgs.themeMap, true);
        eventManager.dispatchEvent("themeinitialize");
    });

    // Register to receive the on theme change event
    themeProxy.addEventListener("themechanged", function (eventArgs) {
        updateTheme(eventArgs.themeMap, false);
        eventManager.dispatchEvent("themechanged");
    });

    function updateTheme(themeMap, isFirst) {
        /// <summary>
        /// Updates css files with the current theme     
        /// </summary>
        /// <param name="themeMap" type="Object">The json theme map of token key value pairs</param>
        /// <param name="isFirst" type="bool">Indicates whether this is the initial theme update</param>

        // Set the 'plugin' token map, process css files and themed images
        tokenMap = themeMap;
        if (!tokenMap) {
            Plugin._logError("JSPlugin.4001");
            return;
        }
        isInitial = isFirst;
        processCssFiles();
        processImages(document);
    }

    function getValue(key) {
        /// <summary>
        /// Retrieves a token value from the token map for a given key     
        /// </summary>
        /// <param name="key" type="String">token key</param>
        /// <returns type="String">token value</returns>

        if (!tokenMap[key]) {
            throw new Error(Plugin.Resources.getErrorString("JSPlugin.4002"));
        }

        return tokenMap[key];
    }

    function processCssFiles() {
        /// <summary>
        ///     Called when a theme change event is fired from the host.  
        ///     Processes the css files so that theme tokens are replaced with the token map values.
        ///     Only links that contain the attribute data-plugin-theme="true" will be processed.  
        /// </summary>

        // Create a new style element and token replace the base css 'plugin' styles
        var pluginStyle = document.createElement("style");
        pluginStyle.type = "text/css";
        pluginStyle.innerHTML = tokenReplaceContents(pluginCss);

        // Insert this style element before all other head children to respect css overrides 
        var firstNode = document.head.firstChild;
        if (firstNode) {
            document.head.insertBefore(pluginStyle, firstNode);
            if (firstNode.id === "pluginCss") {
                // Remove the old 'pluginCss' style node
                document.head.removeChild(firstNode);
            }
        } else {
            document.head.firstChild = pluginStyle;
        }
        // Set the id of the new style node
        pluginStyle.id = "pluginCss";

        // Determine if there are other css theme files to process
        var cssThemeFiles = document.querySelectorAll("[data-plugin-theme='true']");

        // If this is the initial theme change and there are no other
        // files to process notify the host that the theme is ready.
        if (isInitial && cssThemeFiles.length === 0) {
            themeProxy.fireThemeReady();
            return;
        }

        for (var i = 0; i < cssThemeFiles.length; i++) {
            var styleNode = cssThemeFiles[i];
            // Get the source of the file to process
            var href = (!styleNode.href ? styleNode.getAttribute("data-plugin-theme-href") : styleNode.href);
            // If this is the initial theme change event and the final file to process
            var fireThemeReady = (isInitial && (i === cssThemeFiles.length - 1));
            processCssFileContents(href, document, styleNode, fireThemeReady);
        }
    }

    function processCssFileContents(href, targetDoc, refNode, fireThemeReady) {
        /// <summary>
        /// Process a css file so that theme tokens are replaced with the token map values
        /// </summary>
        /// <param name="href" type="String">
        ///     The href of the file to process
        /// </param>
        /// <param name="targetDoc" type="Object">
        ///     The document to add the new style to
        /// </param>
        /// <param name="refNode" type="Object" optional="true">
        ///     The node used to position and name the new style node
        /// </param>
        /// <param name="fireThemeReady" type="bool" optional="true">
        ///    A bool indicating whether to notify that the theme is ready
        /// </param>

        return themeProxy.getCssFile(href).done(function (contents) {
            /// <summary>
            /// Get the css file contents from the host
            /// </summary>
            /// <param name="href" type="String">
            ///     The href of the css file
            /// </param>
            /// <param name="callback" type="Function">
            ///     The callback function to trigger when the host's css file contents is retrieved
            /// </param>

            contents = tokenReplaceContents(contents);

            // Create the new style node
            var newStyle = targetDoc.createElement("style");
            newStyle.setAttribute("data-plugin-theme", "true");
            newStyle.setAttribute("data-plugin-theme-href", href);
            newStyle.type = "text/css";
            newStyle.innerHTML = contents;

            if (refNode) {

                // If the reference node has no parent node the
                // CSS file contents have already been processed.
                if (!refNode.parentNode) {
                    return;
                }

                // Add the new style node before the old style node
                targetDoc.head.insertBefore(newStyle, refNode);

                // Remove the old style node and set the new style node id
                targetDoc.head.removeChild(refNode);
                newStyle.id = refNode.id;
            } else {
                // Add the new style node to the document
                targetDoc.head.appendChild(newStyle);
            }

            if (fireThemeReady) {
                // Notify the host that the 'theme' is ready
                themeProxy.fireThemeReady();
            }

        }, function (e) {

            // On error, still notify the host to reveal the scripted control 
            if (fireThemeReady) {
                themeProxy.fireThemeReady();
            }

            Plugin._logError("JSPlugin.4003\r\n" + e.message + "\r\n" + e.stack);
        });
    }

    function tokenReplaceContents(contents) {
        /// <summary>
        /// Token replace a css file contents so that theme tokens are replaced with the token map values     
        /// </summary>
        /// <param name="contents" type="String">The css file contents to token replace</param>
        /// <returns type="String">The token replaced css file contents</returns>

        return contents.replace(declarationRegEx, function (declaration, indent, property, defaultValue, replacer, suffix) {
            // 'declaration' - matches the css declaration property name and value pair with 'replacer' comments
            // 'indent' - matches any white space before the css property so we keep it readable after transformation
            // 'property' - matches characters before a colon (:) to give the property name
            // 'defaultValue' - matches characters after the colon, up until the semicolon (;) to give the property's default value
            // 'replacer' - matches the special comment style which will be used to replace the css value with the host's theme info.
            // 'suffix' - matches any characters after the token 'replacer'.

            // Keep track of the token replacement count
            var replaceCount = 0;

            // Replace the tokens with token map values
            var newValue = replacer.replace(tokenRegEx, function (tokenMatch, token) {
                // 'tokenMatch' - matches the full token and curly brackets ({token})
                // 'token' - matches the token without the curly brackets.
                replaceCount++;
                return tokenMap[token];
            });

            // If the new value contains undefined or null,  or no tokens have been replaced, use the default value instead
            if (newValue.match(undefinedRegEx) || replaceCount === 0) {
                newValue = defaultValue;
            }
            return indent + property + ": " + newValue + ";" + suffix;
        });
    }

    function processImages(targetDoc) {
        /// <summary>
        ///     Called when a theme change event is fired from the host.  
        ///     Processes images by replacing the image src with a data URI from the token map. 
        ///     Only images that contain the attribute 'data-plugin-theme-src' will be processed.   
        /// </summary>
        /// <param name="targetDoc" type="Object">
        ///     The document that contains the themed images to process
        /// </param>

        var images = targetDoc.querySelectorAll("[data-plugin-theme-src]");
        for (var i = 0; i < images.length; i++) {
            images[i].src = getValue(images[i].getAttribute("data-plugin-theme-src"));
        }
    }

    // Return public functions
    Plugin.Theme = {
        getValue: getValue,
        addEventListener: eventManager.addEventListener.bind(eventManager),
        removeEventListener: eventManager.removeEventListener.bind(eventManager)
    };

    Plugin.Theme._cssHelpers = Plugin.Theme._cssHelpers || {};
    Plugin.Theme._cssHelpers.processCssFileContents = processCssFileContents;
    Plugin.Theme._cssHelpers.processImages = processImages;
})();

// 'Plugin'.'VS'.Tooltip
(function tooltipInit() {
    "use strict";

	// Verify the host is Visual Studio Native Client
    if (!Plugin.detect.isVisualStudioNativeClient) {
        return;
    }

    Plugin.VS = Plugin.VS || {};

    // Y-offset of the tooltip relative to the top-left corner of the mouse pointer
    var tooltipOffsetY = 15;
    var defaultDelay;

    var tooltipObject = null;
    var tooltipReset = true;
    var scheduledShow;
    var scheduledDismiss;
    var mousePosition = { x: 0, y: 0 };

    function removeAllTooltipStyles(tooltip) {
        /// <summary>
        /// Removes all styles from the tooltip's document head.
        /// </summary>
        /// <param name="tooltip" type="Object">the tooltip from which to remove styles</param>

        var styles = tooltip.document.head.querySelectorAll("style, link[type='text/css']");
        for (var i = 0; i < styles.length; i++) {
            var node = styles[i];
            tooltip.document.head.removeChild(node);
        }
    }

    function resetTooltip(tooltip) {
        /// <summary>
        /// Resets the tooltip, so it can be assigned new styles and content.
        /// </summary>
        /// <param name="tooltip" type="Object">the tooltip to reset</param>

        removeAllTooltipStyles(tooltip);
        tooltip.contentDiv.innerHTML = "";
        tooltip.parent = null;
        tooltipReset = true;
    }

    function dismissTooltip(reset) {
        /// <summary>
        /// Cancels any pending tooltip show operations and hides the active tooltip.
        /// </summary>
        /// <param name="reset" type="Boolean" optional="true">whether to also reset the tooltipObject, defaults to true</param>

        var parent = null;

        if (scheduledShow) {
            clearTimeout(scheduledShow);
            scheduledShow = null;
        }
        if (scheduledDismiss) {
            clearTimeout(scheduledDismiss);
            scheduledDismiss = null;
        }
        if (tooltipObject) {
            parent = tooltipObject.parent;
            tooltipObject.hide();
            if (typeof reset === "undefined" || reset) {
                resetTooltip(tooltipObject);
            }
        }
        
        window.__n("TooltipDismiss", tooltipObject, parent, (typeof reset === "undefined" || reset));
    }

    function dismissTooltipOfParent(element, reset) {
        /// <summary>
        /// Dismiss tooltip if its parent is the given element
        /// </summary>
        /// <param name="element" type="Object">target parent element</param>
        /// <param name="reset" type="Boolean" optional="true">whether to also reset the tooltipObject, defaults to true</param>

        if (tooltipObject && tooltipObject.parent === element) {
            dismissTooltip(reset);
        }
    }

    function applyAllParentStyles(tooltip) {
        /// <summary>
        /// Apply all styles of the current document to the tooltip
        /// </summary>
        /// <param name="tooltip" type="Object">target tooltip object</param>

        // Get all style nodes or css link nodes in the current document head
        var styles = document.head.querySelectorAll("style, link[type='text/css']");
        for (var i = 0; i < styles.length; i++) {
            // Clone the element and apply it to the tooltip
            var node = tooltip.document.createElement(styles[i].nodeName);
            var attributes = styles[i].attributes;
            for (var j = 0; j < attributes.length; j++) {
                if (attributes[j].specified) {
                    node.setAttribute(attributes[j].nodeName, attributes[j].nodeValue);
                }
            }
            node.innerHTML = styles[i].innerHTML;
            tooltip.document.head.appendChild(node);
        }
    }

    function getAbsoluteUri(relUri) {
        /// <summary>
        /// Retrieves an absolute URI from a given relative URI
        /// </summary>
        /// <param name="relUri" type="String">relative URI</param>
        /// <returns type="String">absolute URI</returns>

        var a = document.createElement("a");
        a.setAttribute("href", relUri);
        return a.href;
    }

    function applyCssStyles(tooltip, cssFiles) {
        /// <summary>
        /// Process and apply styles to the tooltip from a list of CSS files
        /// </summary>
        /// <param name="tooltip" type="Object">target tooltip object</param>
        /// <param name="cssFiles" type="Array">CSS files to apply</param>

        for (var i = 0; i < cssFiles.length; i++) {
            // Get the absolute URI for each file and process its contents
            var href = getAbsoluteUri(cssFiles[i]);
            Plugin.Theme._cssHelpers.processCssFileContents(href, tooltip.document);
        }
    }

    function createBlankTooltip(cssFiles) {
        /// <summary>
        /// Creates an empty tooltip with inherited and base tooltip styles
        /// </summary>
        /// <param name="cssFiles" type="Array" optional="true">CSS files to apply</param>
        /// <returns type="Object">tooltip object</returns>

        if (!tooltipReset) {
            throw new Error(Plugin.Resources.getErrorString("JSPlugin.4004"));
        }
       
        var p = tooltipObject;

        if (p === null) {
            p = window.createPopup();

            p.document.addEventListener("mouseover", function () {
                p.hide();
                window.__n("TooltipDismiss", p, p.parent, false);
            });
        }

        // Set the document's language and direction attributes
        var htmlTags = p.document.getElementsByTagName("html");
        if (htmlTags.length > 0) {
            htmlTags[0].dir = Plugin.VS.Culture.dir;
            htmlTags[0].lang = Plugin.VS.Culture.lang;
        }

        if (!p.contentDiv) {
            var pbody = p.document.body;
            p.contentDiv = p.document.createElement("div");
            p.contentDiv.className = "plugin-vs-tooltip";
            pbody.appendChild(p.contentDiv);
        }
        applyAllParentStyles(p);

        if (cssFiles) {
            applyCssStyles(p, cssFiles);
        }

        return p;
    }

    function createNewTooltipFromContent(content, cssFiles) {
        /// <summary>
        /// Creates a new tooltip with the given content and css styles
        /// </summary>
        /// <param name="content" type="String">tooltip content</param>
        /// <param name="cssFiles" type="Array" optional="true">CSS files to apply</param>
        /// <returns type="Object">tooltip object</returns>

        var p = createBlankTooltip(cssFiles);
        var pcontent = p.contentDiv;

        if (pcontent) {
            if (typeof content === "string") {
                // If the content is a string, treat it as HTML content, and set innerHTML.
                pcontent.innerHTML = content;
                // Process VS themed images
                Plugin.Theme._cssHelpers.processImages(pcontent);
            } else if (content) {
                // Otherwise, treat the content as text-only content, and set the innerText.
                pcontent.innerText = content;
            }
        }
        return p;
    }

    function createNewTooltipFromString(contentString, cssFiles) {
        /// <summary>
        /// Creates a new tooltip with the given text-only content and css styles
        /// </summary>
        /// <param name="contentString" type="String">content</param>
        /// <param name="cssFiles" type="Array" optional="true">CSS files to apply</param>
        /// <returns type="Object">tooltip object</returns>

        var p = createBlankTooltip(cssFiles);
        var pcontent = p.contentDiv;

        if (pcontent && contentString) {
            pcontent.innerText = contentString;
        }
        return p;
    }

    function showTooltipImmediate(tooltip, duration, x, y) {
        /// <summary>
        /// Displays the tooltip object 
        /// </summary>
        /// <param name="tooltip" type="Object">tooltip object</param>
        /// <param name="duration" type="Number" optional="true">maximum duration of the tooltip</param>
        /// <param name="x" type="Number" optional="true">x-coordinate of the tooltip</param>
        /// <param name="y" type="Number" optional="true">y-coordinate of the tooltip</param>

        // Calculate the zoom factor
        var zoomFactor = 1;
        if (window.screen) {
            zoomFactor = Math.pow(window.screen.deviceXDPI / window.screen.logicalXDPI, 2);
        }

        if (tooltip) {
            // Default x,y coordinates to the current mouse position
            x = (typeof x === "number") ? x : mousePosition.x;
            var offsetFactor = 0;
            if (typeof y !== "number") {
                y = mousePosition.y;
                offsetFactor = 1;
            }

            // Default duration is 10 times the Double Click time on the system
            duration = (typeof duration === "number") ? duration : ((defaultDelay || (defaultDelay = __hostExternalObject.getDblClickTime())) * 10);

            // Obtain the size of content
            var screenTop = __hostExternalObject.getScreenTop(x, y);
            var screenLeft = __hostExternalObject.getScreenLeft(x, y);
            var screenRight = __hostExternalObject.getScreenRight(x, y);
            var screenBottom = __hostExternalObject.getScreenBottom(x, y);
            var maxWidth = screenRight - screenLeft;
            var maxHeight = screenBottom - screenTop;

            tooltip.show(screenLeft, screenTop, maxWidth, maxHeight);
            // Get the width/height of the content
            var width = tooltip.contentDiv.offsetWidth;
            var height = tooltip.contentDiv.offsetHeight;
            tooltip.hide();

            // Check the calculated width/height
            tooltip.show(screenLeft, screenTop, width, height);
            if (tooltip.contentDiv.offsetHeight > height) {
                // Increment width if the original offset width is insufficient
                width++;
            }
            tooltip.hide();

            // Apply y offset, if any, after sizing
            y += (offsetFactor * tooltipOffsetY);

            // By default tooltip displays to the bottom right of the x,y position. If the tooltip would 
            // intersect a screen edge, translate/crop the tooltip so it fits on the screen.
            var distToScreenRight = screenRight - (x + width);

            // If the tooltip intersects the right of the screen, move the tooltip to the left.
            if (distToScreenRight < 0) {
                // First, try moving the tooltip such that its right edge would be on the screen edge.
                var tempX = x + distToScreenRight;
                var distToScreenLeft = tempX - screenLeft;

                // If this move causes the tooltip to intersect the edge of the screen, adjust the x position
                // and width of the tooltip such that its left edge is on the screen edge and it fits within the screen width.
                if (distToScreenLeft < 0) {
                    x = tempX - distToScreenLeft;
                    width += distToScreenLeft;
                } else {
                    x = tempX;
                }
            }
            var distToScreenBottom = screenBottom - (y + height);

            // If the tooltip intersects the bottom of the screen, move the tooltip up.
            if (distToScreenBottom < 0) {
                // First, try translating the tooltip across the initial y position (before offset), 
                // so it is displayed above, rather than below.
                var tempY = y - ((2 * offsetFactor * tooltipOffsetY * zoomFactor) + height);
                var distToScreenTop = tempY - screenTop;

                // If there is more room on the screen above the initial y position, accept the "above" positioning.
                if (distToScreenTop > distToScreenBottom) {
                    y = tempY;

                    // If this positioning intersects the top of the screen, adjust the y position and height of the tooltip, 
                    // such that its top edge is on the screen edge and its bottom edge remains in the same position.
                    if (distToScreenTop < 0) {
                        y -= distToScreenTop;
                        height += distToScreenTop;
                    }
                } else {
                    // If there is more room below the initial y position, adjust the height so the bottom edge is on the screen edge.
                    height += distToScreenBottom;
                }
            }

            // Adjust final x and y values for zoom factor
            x = Math.floor(x / zoomFactor);
            y = Math.floor(y / zoomFactor);

            tooltip.show(x, y, width, height);
            scheduledShow = null;

            // If a duration is specified, schedule the tooltip to be dismissed after that time.
            if (duration > 0) {
                scheduledDismiss = setTimeout(function () {
                    dismissTooltip(false);
                    scheduledDismiss = null;
                }, duration);
            }

            window.__n("TooltipShow", tooltip, x, y, width, height, duration, scheduledDismiss);
        }
    }

    function scheduleShowTooltip(tooltip, delay, duration, x, y) {
        /// <summary>
        /// Schedule a tooltip to be shown
        /// </summary>
        /// <param name="tooltip" type="Object">tooltip object</param>
        /// <param name="delay" type="Number" optional="true">delay in milliseconds of tooltip show, defaults to system Double Click time</param>
        /// <param name="duration" type="Number" optional="true">duration of tooltip show in milliseconds</param>
        /// <param name="x" type="Number" optional="true">x-coordinate of the tooltip</param>
        /// <param name="y" type="Number" optional="true">y-coordinate of the tooltip</param>
        /// <returns type="Number">the indentifier of the timeout operation</returns>

        if (!tooltip) { return null; }

        // Set default delay
        delay = (typeof delay === "number") ? delay : (defaultDelay || (defaultDelay = __hostExternalObject.getDblClickTime()));

        // If there is no delay, show the tooltip immediately
        if (delay <= 0) {
            showTooltipImmediate(tooltip, duration, x, y);
            return null;
        }

        // Otherwise, schedule the tooltip to be shown after delay
        var timeout = setTimeout(function () {
            showTooltipImmediate(tooltip, duration, x, y);
        }, delay);
        
        window.__n("TooltipShowScheduled", tooltip, delay);

        return timeout;
    }

    function showTooltip(config, parent) {
        /// <summary>
        /// Creates and displays a new tooltip with the given configuration
        /// </summary>
        /// <param name="config" type="Object">configuration of the tooltip</param>
        /// <param name="parent" type="Object" optional="true">parent element</param>

        // Dismiss any active tooltip
        dismissTooltip();

        var tooltip = null;
        var options = {};
        if (config && typeof config === "object") {
            var tooltipContent;
            if (config.resource) {
                if (config.content || config.content === "") {
                    // If there is a content property to fall back to, ignore errors.
                    try {
                        tooltip = createNewTooltipFromString(Plugin.Resources.getString(config.resource), config.cssFiles);
                    } catch (e) {}
                } else {
                    tooltip = createNewTooltipFromString(Plugin.Resources.getString(config.resource), config.cssFiles);
                }
            }
            if (!tooltip && (config.content || config.content === "")) {
                tooltip = createNewTooltipFromContent(config.content, config.cssFiles);
            }

            if (!tooltip) {
                throw new Error(Plugin.Resources.getErrorString("JSPlugin.4005"));
            }

            options = config;
        } else {
            tooltip = createNewTooltipFromString(config, null);
        }

        if (!tooltip) {
            throw new Error(Plugin.Resources.getErrorString("JSPlugin.4006"));
        }

        tooltip.parent = parent;
        tooltipObject = tooltip;
        tooltipReset = false;

        scheduledShow = scheduleShowTooltip(tooltip, options.delay, options.duration, options.x, options.y);
    }

    function initializeElementTooltip(element) {
        /// <summary> 
        /// Initialize tooltip for the given element, based on its data-plugin-vs-tooltip attribute.
        /// </summary>
        /// <param name="element" type="Object">the HTML element for which to initialize a tooltip</param>

        if (!element || !element.addEventListener) {
            Plugin._logError("JSPlugin.4007");
            return;
        }

        // Helper function for determining if an element is a child of another.
        function hasChild(element, childCandidate) {
            var currentParent = childCandidate ? childCandidate.parentNode : null;

            while (currentParent && currentParent !== document.body) {
                if (currentParent === element) {
                    return true;
                }
                currentParent = currentParent.parentNode;
            }

            return false;
        }

        // Show tooltip on 'mouseover'
        function onMouseOver(e) {
            if (!e.currentTarget.hasAttribute("data-plugin-vs-tooltip")) {
                // If this element no longer has a 'data-plugin-vs-tooltip' attribute, un-initialize it.
                e.currentTarget.removeEventListener("mouseover", onMouseOver);
                e.currentTarget.removeEventListener("mouseout", onMouseOut);
                e.currentTarget.removeEventListener("mousedown", onMouseDown);
                e.currentTarget.__plugin_tooltip_initialized = false;
                return;
            }

            // If the tooltip for this element is already active, the target element's own tooltip is already 
            // active (the most specific tooltip), or a tooltip belonging to a child element
            // that is also a parent of the target element is active (a more specific tooltip), do not 
            // show this tooltip.
            if (tooltipObject && !tooltipReset && tooltipObject.parent &&
                ((tooltipObject.parent === e.currentTarget) || (tooltipObject.parent === e.target) ||
                (hasChild(tooltipObject.parent, e.target) && hasChild(e.currentTarget, tooltipObject.parent)))) {
                return;
            }

            // Create the tooltip based on the value of thee element's 'data-plugin-vs-tooltip' attribute
            var tooltipConfigStr = e.currentTarget.getAttribute("data-plugin-vs-tooltip");

            var config;
            if ((typeof tooltipConfigStr === "string") && (tooltipConfigStr.length > 0) && (tooltipConfigStr[0] === "{")) {
                config = JSON.parse(tooltipConfigStr);
            } else {
                config = tooltipConfigStr;
            }

            showTooltip(config, e.currentTarget);
        };

        // Listen to this event on capture, so child tooltips will take precedence over parent tooltips
        element.addEventListener("mouseover", onMouseOver, true);

        // Dismiss tooltip on 'mouseout' only if the tooltip being dismissed belongs to this parent
        function onMouseOut(e) {
            // Only take action if the relatedTarget is not the currentTarget or a child of the currentTarget
            if (e.relatedTarget && (e.currentTarget !== e.relatedTarget) && !hasChild(e.currentTarget, e.relatedTarget)) {
                dismissTooltipOfParent(e.currentTarget);
            }
        };
        element.addEventListener("mouseout", onMouseOut);

        // Dismiss tooltip on 'mousedown' only if the tooltip being dismissed belongs to this parent
        function onMouseDown(e) {
            dismissTooltipOfParent(e.currentTarget, false);
        };
        element.addEventListener("mousedown", onMouseDown);

        element.__plugin_tooltip_initialized = true;
    }

    document.addEventListener("DOMContentLoaded", function () {
        // Get all elements with the 'data-plugin-vs-tooltip' attribute and initialize them
        var withTooltipData = document.querySelectorAll("[data-plugin-vs-tooltip]");
        for (var i = 0; i < withTooltipData.length; i++) {
            initializeElementTooltip(withTooltipData[i]);
        }
    }, false);

    document.addEventListener("mouseout", function (e) {
        // Dismiss the active tooltip when the mouse leaves the document
        if (!e.relatedTarget || e.relatedTarget.nodeName === "HTML") {
            dismissTooltip();
        }
    }, false);

    document.addEventListener("mouseover", function (e) {
        var tooltipConfig;
        if (!e.target.__plugin_tooltip_initialized && e.target.hasAttribute("data-plugin-vs-tooltip")) {
            initializeElementTooltip(e.target);
        }
    }, true);

    document.addEventListener("mousemove", function (e) {
        // Track the mouse position for positioning tooltips
        mousePosition.x = e.screenX;
        mousePosition.y = e.screenY;
    }, false);

    // Public Functions
    Plugin.VS.Tooltip = {
        show: function (config) {
            /// <signature>
            /// <summary> 
            ///     Show a tooltip with given configuration
            /// </summary>
            /// <param name="config" type="Object">
            ///     The configuration object describing the tooltip to show.<br/>
            ///     Must include at least one of the following properties:<br/>
            ///         'content' - String that contains the HTML content of the tooltip, used as default
            ///             if 'resource' is not provided or cannot be found<br/>
            ///         'resource' - String identifying the resource string to include as content<br/>
            ///     Other optional properties include:<br/>
            ///         'delay' - Number of milliseconds to delay before showing the tooltip<br/>
            ///         'duration' - maximum Number of milliseconds the tooltip should be displayed (0 is no maximum)<br/>
            ///         'x' - x-coordinate, relative to the screen, at which to show the tooltip<br/>
            ///         'y' - y-coordinate, relative to the screen, at which to show the tooltip<br/>
            ///         'cssFiles' - Array of Strings, representing css file paths to apply as additional tooltip styles
            /// </param>
            /// </signature>
            /// <signature>
            /// <summary> 
            ///     Show a tooltip with given text content
            /// </summary>
            /// <param name="config" type="String">string content to display in a text-only tooltip</param>
            /// </signature>
            showTooltip(config, null);
        },
        dismiss: dismissTooltip,
        initializeElementTooltip: initializeElementTooltip
    };

})();

// 'Plugin'.'Diagnostics'
(function DiagnosticsInit() {
    "use strict";

    Plugin.Diagnostics = Plugin.Diagnostics || {};

    var onErrorHandler = function (message, uri, lineNumber) {
        /// <summary> 
        /// Generates a WER report for the error and terminates the sandbox process
        /// </summary>
        /// <param name="message" type="String">Error message</param>
        /// <param name="uri" type="String">Uri where error was occurred</param>
        /// <param name="lineNumber" type="String">Line number where error was occurred</param>
        Plugin.Diagnostics.reportError(message, uri, lineNumber);
        Plugin.Diagnostics.terminate();
        return true;
    };

    Plugin.Diagnostics.reportError = function (error, uri, lineNumber, additionalInfo) {
        /// <signature>
        /// <summary>
        /// Generates a WER report from an error object
        /// </summary>
        /// <param name="error" type="Error">Error object</param>
        /// <returns type="Number">0 for success, positive value for partial success (able to report only some information), negative value for failure</returns>
        /// </signature>

        /// <signature>
        /// <summary> 
        /// Generates a WER report with the information provided and queues it for submission to Watson server
        /// </summary>
        /// <param name="message" type="String">Error message</param>
        /// <param name="uri" type="String">Uri where error was occurred</param>
        /// <param name="lineNumber" type="String"> Line number where error was occurred</param>
        /// <param name="additionalInfo" type="String" optional="true">Any additional information about the error</param>
        /// <returns type="Number">0 for success, positive value for partial success (able to report only some information), negative value for failure</returns>
        /// </signature>

        var message;

        if (error instanceof Error) {
            message = error.message ? error.message.toString() : null;
            uri = null;
            lineNumber = null;
            additionalInfo = "Error number: " + error.number;
            additionalInfo += "\r\nStack: " + error.stack;
        } else {
            message = error ? error.toString() : null;
            uri = uri ? uri.toString() : null;
            lineNumber = lineNumber ? lineNumber.toString() : null;
            additionalInfo = additionalInfo ? additionalInfo.toString() : null;
        }

        return window.__hostExternalObject.reportError(message, uri, lineNumber, additionalInfo);
    };

    Plugin.Diagnostics.terminate = function () {
        /// <summary>Terminates the sandbox process by raising a FailFastException</summary>
        window.__hostExternalObject.terminate();
    };

    window.onerror = onErrorHandler;
})();


(function hostProxyInit() {

    Plugin.Host = Plugin.Host || {};

    // Attach to the'Plugin'.'Host' host object	
    Plugin.Host._hostProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.Host", {
        showDocument: function (documentPath, line, col) {
            return this._call("showDocument", documentPath, line, col);
        },
    }, true);

})();

(function hostInit() {
    "use strict";

    // Return public functions
    Plugin.Host.showDocument = function (documentPath, line, col) {
        /// <summary>
        /// Calls VS to have them opens a document and set the caret at a specific row and col.
        /// </summary>
        /// <param name="documentPath" type="String">The full document path to open</param>
        /// <param name="line" type="String">The line number to set the caret on (1-based)</param>
        /// <param name="col" type="String">The col number to set the caret on (1-based)</param>

        return Plugin.Host._hostProxy.showDocument(documentPath, line, col);
    };

})();
// 'Plugin'.VS.Culture
(function vsKeyboard() {
    "use strict";

    // Verify the host is Visual Studio Native Client
    if (!Plugin.detect.isVisualStudioNativeClient) {
        return;
    }

    Plugin.VS = Plugin.VS || {};
    Plugin.VS.Keyboard = Plugin.VS.Keyboard || {};

    var clipboardGroup = 0;
    var zoomGroup = 1;

    var zoomState = true;

    function disableMouseWheelZoom(e) {
        if (e.ctrlKey) {
            e.preventDefault();
        }
    }

    Plugin.VS.Keyboard.setClipboardState = function setClipboardState(state) {
        /// <summary>   
        /// Set whether the browser will try to handle clipboard keys or allow 
        /// Visual Studio to process them.
        /// </summary>
        /// <param name="state" type="Boolean">
        /// True indicates the browser should handle the clipboard keys.
        /// False, indicates the browser should ignore the clipboard keys 
        /// allowing Visual Studio to see them.
        /// </param>

        window.external.setHotKeysState(clipboardGroup, !!state);
    };

    Plugin.VS.Keyboard.setZoomState = function setZoomKeyState(state) {
        /// <summary>   
        /// Set whether the user should be allowed to control the zoom
        /// state of the browser through hot keys, Ctrl -, Ctrl +, Ctrl 0,
        /// and the mouse wheel, or should zooming be disabled.
        /// </summary>
        /// <param name="state" type="Boolean">
        /// True indicates the user should be allowed to zoom the browser.
        /// False indicates the user should not be allowed to zoom the browser.
        /// </param>

        state = !!state;
        if (zoomState !== state) {
            window.external.setHotKeysState(zoomGroup, state);
            if (!state) {
                window.addEventListener("mousewheel", disableMouseWheelZoom, false);
            } else {
                window.removeEventListener("mousewheel", disableMouseWheelZoom);
            }
            zoomState = state;
        }
    };
})();

(function solutionInit() {
    "use strict";

	// Verify the host is Visual Studio Native Client
    if (!Plugin.detect.isVisualStudioNativeClient) {
        return;
    }

    Plugin.VS = Plugin.VS || {};
	Plugin.VS.Experimental = Plugin.VS.Experimental || {};
	Plugin.VS.Experimental.Solution = Plugin.VS.Experimental.Solution || {};

    // Attach to the'Plugin'.'VS'.'Experimental'.'Solution' host object	
    Plugin.VS.Experimental.Solution._hostProxy = Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Plugin.VS.Experimental.Solution", {

        getItems: function (id, previousChildId, count, levels) {
            return this._call("getItems", {
				id: id, 
				previousChildId: previousChildId, 
				count: count, 
				levels: levels
			});
        },

        getItemProperties: function (id) {
            return this._call("getItemProperties", id);
        }

    }, true);

    // Return public functions
    Plugin.VS.Experimental.Solution.getItems = function (id, previousChildId, count, levels) {
        /// <summary>
        /// Returns a promise that will have items in the IVsHierarchy that the caller can 
		/// use to iterate over the project.
        /// </summary>
        /// <param name="id" type="String">The id of the root node to iterate over.  Empty means the top of the solution.</param>
        /// <param name="previousChildId" type="String">The id of the child item to start iterating from.  Empty means from the root.</param>
        /// <param name="count" type="int">The number of items to return.</param>
		/// <param name="level" type="int">The number of levels deep into the hierarchy the host should traverse.</param>

        return Plugin.VS.Experimental.Solution._hostProxy.getItems(id, previousChildId, count, levels);
    };

    Plugin.VS.Experimental.Solution.getItemProperties = function (id) {
        /// <summary>
        /// Returns a promise that will contain the properties of a particular item.
        /// </summary>
        /// <param name="id" type="String">The id of the node to find properties on.</param>
        
        return Plugin.VS.Experimental.Solution._hostProxy.getItemProperties(id);
    };

    // Expose events on the host proxy:
    Plugin.VS.Experimental.Solution.addEventListener = Plugin.VS.Experimental.Solution._hostProxy.addEventListener.bind(Plugin.VS.Experimental.Solution);
    Plugin.VS.Experimental.Solution.removeEventListener = Plugin.VS.Experimental.Solution._hostProxy.addEventListener.bind(Plugin.VS.Experimental.Solution);

})();
// 'Plugin'.VS.Utilities
(function vsUtilitiesInit() {
    "use strict";

    // Verify the host is Visual Studio Native Client
    if (!Plugin.detect.isVisualStudioNativeClient) {
        return;
    }

    Plugin.VS = Plugin.VS || {};
    Plugin.VS.Utilities = Plugin.VS.Utilities || {};
    Plugin.VS.Utilities.createExternalObject = window.__hostExternalObject.createExternalObject;
})();



// SIG // Begin signature block
// SIG // MIIaiQYJKoZIhvcNAQcCoIIaejCCGnYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFITyyMsmWNwV
// SIG // il/+1mgXUh3u9iNhoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggR8MIIEeAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGeMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBQMQYgas9iUm6u6+EZ/DLwR
// SIG // 7yadzDA+BgorBgEEAYI3AgEMMTAwLqAUgBIAcABsAHUA
// SIG // ZwBpAG4ALgBqAHOhFoAUaHR0cDovL21pY3Jvc29mdC5j
// SIG // b20wDQYJKoZIhvcNAQEBBQAEggEAStQkWAL9lrTXCKRH
// SIG // kUjru0MJzSq5rxQyR+ANkzGbwvw7g9Y91KDsGde2qpdf
// SIG // UInysFLuaY13+vsE0zHGetUGaJeE2QQMS8tBnyOtXtm6
// SIG // ZB7Aiw5mBEaEhbjOLA3F0CCmMW295+QfWR/rNKnb6QNl
// SIG // MAtXfgayyE5IGcN2j5Qf5y3IKrrV9FJxC0oZ2YdgC1n8
// SIG // QdKy1U8oWWaj935BxwHx2+5ZmvSo5NS99FDcINGqaK8T
// SIG // VKtP4GKR6AttIlFkOPMqQip6KTQwSaN5YQ1qo4jeGlQb
// SIG // Cvq+m/M95W5bJrxqsud4zlFFYvZoHa08GH5q9Xwdarcx
// SIG // mGK8j21Nq3LIWNQBRqGCAh8wggIbBgkqhkiG9w0BCQYx
// SIG // ggIMMIICCAIBATCBhTB3MQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQ
// SIG // Q0ECCmECjkIAAAAAAB8wCQYFKw4DAhoFAKBdMBgGCSqG
// SIG // SIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkF
// SIG // MQ8XDTEzMDMxNTA2MzM1NVowIwYJKoZIhvcNAQkEMRYE
// SIG // FE22eD9NR24FOfY0v5LRTjwO2vy3MA0GCSqGSIb3DQEB
// SIG // BQUABIIBAIB+TLL3YIUTgim5TMQA0HhMQhc/g6MRmr6s
// SIG // 00i5a2fJXewBn794ey72fDlzGewm9sDp/nfPYI6LS25f
// SIG // 9/diCAL9UMiEJ3G1+hy6hmcbqdcYg8FlUVKlJfY4XtJB
// SIG // 6B+ysxpHzxRxERHQXwr/Np6TVNoIcgnNmUC2AAasdfMM
// SIG // rcIjipEYe6VATlTjPAXbspqLrzSGRg3ralEAGWx3vHLZ
// SIG // 6nOaMnRM2wJmEXwW7aBVWiw376ZmqMbX8Gytow/g69mc
// SIG // zVRbLJqXwsUeOfmIlCVgHKH/FZymRGTe6RCulmHHN+y/
// SIG // 7qUbjB/PFQNFSPdyouePKeiD/ojYQ7iei4ZeMiPZ0i4=
// SIG // End signature block
